import os
import logging
import requests
from retry import retry

import google.protobuf.text_format as text_format

import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa
import search.plutonium.deploy.proto.rescan_pb2 as rescan  # noqa
import resource_pool


# todo: extra catch http exceptions
def notify_resource(resource, spec, resources):
    action_type = spec.WhichOneof('Action')
    if action_type == 'HttpRequest':
        if spec.Extended:
            return _notify_extended(
                spec.HttpRequest.Method,
                spec.HttpRequest.Port,
                spec.HttpRequest.Path,
                resources,
                spec.NotificationCachePath or None,
            )
        else:
            ok = _notify_http(spec.HttpRequest.Method, spec.HttpRequest.Port, spec.HttpRequest.Path)
            return {resource: ok}
    else:  # Resource with empty notification is never active.
        return {resource: False}


# todo: extra catch http exceptions
def rescan_resources(spec, resource_targets, storage):
    return _notify_rescan(
        spec.HttpRequest.Method,
        spec.HttpRequest.Port,
        spec.HttpRequest.Path,
        resource_targets,
        storage,
        spec.NotificationCachePath or None,
    )


@retry(tries=4, delay=1, backoff=2)
def _notify_http(method, port, path):
    return requests.request(method, 'http://localhost:{}{}'.format(port, path), timeout=5).ok


def _notify_extended(method, port, path, resources, notification_cache_path=None):
    notification = tables.TExtendedNotification(
        Resources=[tables.TExtendedNotification.TResource(Namespace=resource.namespace, LocalPath=resource.name)
                   for resource in resources])

    data = text_format.MessageToString(notification)

    if notification_cache_path is not None:
        _save_notification_cache(data, notification_cache_path)

    response = requests.request(method, 'http://localhost:{}{}'.format(port, path), data=data, timeout=5)
    status = text_format.Parse(response.text, tables.TExtendedStatus())

    _store_custom_status(status.DaemonStatus)

    result = {}
    for resource in status.Resources:
        key = resource_pool.ResourceKey(namespace=resource.Namespace, name=resource.LocalPath, rbtorrent=None, size=0)
        result[key] = resource.Valid
    return result


def _notify_rescan(method, port, path, resource_targets, storage, notification_cache_path=None):
    notification = rescan.TExtendedNotification(
        Resources=[
            rescan.TResourceNotification(
                Namespace=resource.namespace,
                LocalPath=resource.name,
                AbsolutePath=storage.link_path(resource),
                Labels=target.ResourceLabels
            )
            for resource, target in resource_targets.iteritems()
        ]
    )

    data = text_format.MessageToString(notification)

    if notification_cache_path is not None:
        _save_notification_cache(data, notification_cache_path)

    response = requests.request(method, 'http://localhost:{}{}'.format(port, path), data=data, timeout=5)
    status = text_format.Parse(response.text, rescan.TExtendedStatus())

    _store_custom_status(status.DaemonStatus)

    result = {}
    for resource in status.Resources:
        key = resource_pool.ResourceKey(namespace=resource.Resource.Namespace, name=resource.Resource.LocalPath,
                                        rbtorrent=None, size=0)
        result[key] = resource.Valid
    return result


def _store_custom_status(data):
    if data:
        _log.debug('Daemon status (first 100 bytes): "%s"', data[:100])


def _save_notification_cache(data, notification_cache_path):
    try:
        swp_notification_cache_path = '{}.swp'.format(notification_cache_path)
        with open(swp_notification_cache_path, 'w') as wfile:
            wfile.write(data)
            os.fsync(wfile)
        os.rename(swp_notification_cache_path, notification_cache_path)

        _log.info('Notification cache saved to "{}"'.format(notification_cache_path))
    except (IOError, OSError):
        _log.error('can\'t save notification cache')


_log = logging.getLogger(__name__)
