import datetime
import logging

import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa
import gevent.threading

import resource as resource_types


_READY = {tables.EDownloadState.ACTIVE, tables.EDownloadState.PREPARED}


class _ResourceState(object):
    def __init__(self, status=tables.EDownloadState.IDLE):
        self.status = status
        self.annotation = None


class _ResourceInfo(object):
    def __init__(self, target, state, niceness=0):
        self.target = target
        self.state = state
        self.niceness = niceness
        self.attempts = 0
        self.last_attempt = datetime.datetime.now()


ResourceKey = resource_types.ResolvedResource


def synchronized(method):
    def wrapped(self, *args, **kwargs):
        with self._lock:
            return method(self, *args, **kwargs)  # noqa
    return wrapped


class ResourcePool(object):
    def __init__(self):
        self._resources = {}  # Dict: ResourceKey -> _ResourceInfo[TPodTarget, _ResourceState]
        self._lock = gevent.threading.Lock()
        self._resource_counter = 0
        self._reset = False

    @synchronized
    def __len__(self):
        return len(self._resources)

    @synchronized
    def __contains__(self, item):
        return item in self._resources

    @synchronized
    def ready_count(self):
        return sum(1 for info in self._resources.values() if info.state.status in _READY)

    @synchronized
    def not_ready_count(self):
        return sum(1 for info in self._resources.values() if info.state.status not in _READY)

    @synchronized
    def add_existing(self, key, target):
        self._resources[key] = _ResourceInfo(target=target,
                                             state=_ResourceState(status=tables.EDownloadState.PREPARED))

    @synchronized
    def keys(self):
        return self._resources.keys()

    @synchronized
    def get_ready(self):
        return {resource: info.target for resource, info in self._resources.iteritems()
                if info.state.status in _READY}

    @synchronized
    def reset(self, targets):
        for key, value in targets.iteritems():
            assert isinstance(key, ResourceKey)
            assert isinstance(value, tables.TPodTarget)

        resources = {}
        for resource, target in targets.iteritems():
            if resource in self._resources:
                resources[resource] = self._resources[resource]
            else:
                resources[resource] = _ResourceInfo(target=target, state=_ResourceState(), niceness=self._resource_counter)
                _log.info('Added new resource %s: priority=%d, niceness=%d',
                          resource, target.DownloadPolicy.Priority, self._resource_counter)
                self._resource_counter += 1
        self._resources = resources
        self._reset = True

    @synchronized
    def set_status(self, resource, status):
        assert isinstance(resource, ResourceKey)

        resource_info = self._resources.get(resource)
        if resource_info:
            if resource_info.state.status != status:
                if status == tables.EDownloadState.DOWNLOADING:
                    resource_info.attempts += 1
                    resource_info.last_attempt = datetime.datetime.now()

                _log.debug('Change status [%s]: %s -> %s, attempt %s',
                           resource, resource_info.state.status, status, resource_info.attempts)
                resource_info.state.status = status
        else:
            # It's ok in general.
            _log.debug('Attempt to change status of removed resource %s', resource)

    @synchronized
    def get_status(self, resource):
        if resource in self._resources:
            return self._resources[resource].state.status
        return None

    @synchronized
    def get_statuses(self):
        return {resource: info.state.status for resource, info in self._resources.iteritems()}

    @synchronized
    def list(self):
        resource_sort_key = lambda kv: (-kv[1].target.DownloadPolicy.Priority, kv[1].niceness)
        sorted_resources = ((resource, info.target) for resource, info in sorted(self._resources.iteritems(),
                                                                                 key=resource_sort_key))
        for resource, info in sorted_resources:
            yield resource, info  # Process at least one resource before resetting.

            if self._reset:
                self._reset = False
                break


_log = logging.getLogger(__name__)
