import datetime
import logging
import os

import gevent
import gevent.event
import google.protobuf.text_format as text_format
import pytz

import infra.callisto.deploy.storage.flat_storage as flat_storage
import infra.callisto.deploy.storage.tree_storage as tree_storage
import infra.callisto.protos.deploy.tables_pb2 as tables  # noqa

import download
import resource as deploy_resource


class Storage(object):
    def __init__(self, root_path, cleanup_cooldown, deferred_remove_iterations=0):
        self._root = root_path
        self._flat = flat_storage.FlatStorage(os.path.join(root_path, 'flat'))
        self._tree = tree_storage.LinksTree(os.path.join(root_path, 'tree'),
                                            tree_storage.NamingScheme('__resources__', ''))
        self._cleanup_cooldown = cleanup_cooldown
        self._current_resource = None
        self._deferred_removes = DeferredRemovesStorage(deferred_remove_iterations)

    def root(self):
        return os.path.realpath(self._root)

    def total_space(self):
        if os.path.exists(self._root):
            vfs = os.statvfs(self._root)
            return vfs.f_blocks * vfs.f_frsize
        else:
            return None

    def free_space(self):
        if os.path.exists(self._root):
            vfs = os.statvfs(self._root)
            return vfs.f_bavail * vfs.f_frsize
        else:
            return None

    @property
    def current_resource(self):
        return self._current_resource

    def load_description(self, key):
        return text_format.Parse(self._flat.description(_flat_name(key)), tables.TDeployerResourceDescription())

    def list_ready(self):
        ready = []
        for flat_name in self._flat.list():
            if self._flat.is_fetched(flat_name):
                try:
                    ready.append(self._flat_name_to_resource(flat_name))
                except text_format.ParseError:
                    pass
        return ready

    def is_resource_ready(self, resource):
        flat_name = _flat_name(resource)
        if self._flat.is_fetched(flat_name):
            try:
                stored_resource = self._flat_name_to_resource(_flat_name(resource))
            except text_format.ParseError:
                return False
            if stored_resource.rbtorrent:
                return resource.rbtorrent == stored_resource.rbtorrent
            else:
                return True
        return False

    def invalidate(self, resource):
        """Note that invalidate does not remove the resource"""
        self._flat.remove_description(_flat_name(resource))

    # noinspection DuplicatedCode
    def cleanup_except(self, keep_resources):
        keep_flatnames = {_flat_name(resource) for resource in keep_resources}
        self._deferred_removes.add_generation(self._flat.list(), keep_flatnames)
        generation = self._deferred_removes.get_oldest_discardable()
        while generation is not None:
            for flat_name in generation:
                self._remove_by_flat_name(flat_name, keep_flatnames, keep_resources)
            self._deferred_removes.discard_oldest()
            generation = self._deferred_removes.get_oldest_discardable()

    def _remove_by_flat_name(self, flat_name, keep_flatnames, keep_resources):
        if flat_name not in keep_flatnames:
            if not self._flat.is_fetched(flat_name):
                _log.info('cleanup: removing incomplete [%s]', flat_name)
                self._flat.remove(flat_name)
            else:
                _log.info('cleanup: removing complete [%s]', flat_name)
                try:
                    resource_ = self._flat_name_to_resource(flat_name)
                except (ValueError, RuntimeError, text_format.ParseError) as err:
                    _log.error('unable to parse description of [{}], error: {}'.format(flat_name, repr(err)))
                else:
                    if resource_ not in keep_resources:
                        self._tree.unset_link(resource_)
                        self._flat.remove(flat_name)
            if self._cleanup_cooldown:
                # avoid disks flooding on mass cleanup
                gevent.sleep(self._cleanup_cooldown)

    def download(self, target, resource):
        """Download protobuf-defined source"""
        with self._flat.add(_flat_name(resource), _dump_description(target)) as destination:
            self._current_resource = resource
            try:
                download.download_resource(self, target.Namespace, target.ResourceSpec, destination)
                self._tree.set_link(resource, destination)
            finally:
                self._current_resource = None

    def realpath(self, resource):
        return self._flat.path_to_resource(_flat_name(resource))

    def link_path(self, resource):
        return self._tree.resource_path(resource)

    def _flat_name_to_resource(self, flat_name):
        target = text_format.Parse(self._flat.description(flat_name), tables.TDeployerResourceDescription()).Target
        return deploy_resource.ResolvedResource(namespace=target.Namespace, name=target.LocalPath,
                                                rbtorrent=None, size=0)


class DeferredRemovesStorage(object):
    def __init__(self, max_generations=0):
        self._max_generations = max_generations
        self._generations = []
        self._all_removes = set()

    def add_generation(self, generation, exclude_names):
        filtered = []
        for name in generation:
            if (name not in exclude_names) and (name not in self._all_removes):
                _log.info('cleanup: deferred remove of [%s]', name)
                self._all_removes.add(name)
                filtered.append(name)
        self._generations.append(filtered)

    def get_oldest_discardable(self):
        if len(self._generations) > self._max_generations:
            return self._generations[0]
        return None

    def discard_oldest(self):
        if len(self._generations) > self._max_generations:
            for name in self._generations[0]:
                self._all_removes.remove(name)
            self._generations.pop(0)


def _flat_name(resource):
    return _safe_join_path(
        resource.namespace,
        '__resources__',
        resource.name
    ).strip('/').replace('/', '_')


def _dump_description(target):
    now = datetime.datetime.now(tz=pytz.timezone('Europe/Moscow'))
    meta = tables.TDeployerResourceDescription.TMeta(Timestamp=now.isoformat())
    description = tables.TDeployerResourceDescription(Target=target, Meta=meta)
    return text_format.MessageToString(description)


# Copypasted from deploy.resource.
# noinspection DuplicatedCode
def _safe_join_path(*parts):
    normalized_parts = []
    first = True
    for part in parts:
        part = part.rstrip('/') if first else part.strip('/')
        if part:
            normalized_parts.append(part)
        first = False
    return os.path.join(*normalized_parts)


_log = logging.getLogger(__name__)
