from __future__ import absolute_import
import os
import json
import collections


class Resource(collections.namedtuple('Resource', [
    'namespace',
    'name',
])):
    def __new__(cls, namespace, name, **kwargs):
        instance = super(cls, cls).__new__(
            cls,
            normalize_namespace(namespace),
            normalize_resource_name(name),
        )

        return instance

    def to_dict(self):
        return {
            'namespace': self.namespace,
            'name': self.name,
        }

    @classmethod
    def from_dict(cls, dict_):
        return Resource(
            dict_['namespace'],
            dict_['name'],
        )


class ResolvedResource(collections.namedtuple('ResolvedResource', [
    'namespace',
    'name',
    'rbtorrent',
    'size',
])):
    def __new__(cls, namespace, name, rbtorrent, size, **kwargs):
        instance = super(cls, cls).__new__(
            cls,
            normalize_namespace(namespace),
            normalize_resource_name(name),
            rbtorrent,
            size,
        )

        return instance

    def get_resource(self):
        return Resource(self.namespace, self.name)

    def to_dict(self):
        return {
            'namespace': self.namespace,
            'name': self.name,
            'rbtorrent': self.rbtorrent,
            'size': self.size,
        }

    @classmethod
    def from_dict(cls, dict_):
        return cls(
            dict_['namespace'],
            dict_['name'],
            dict_.get('rbtorrent'),
            dict_.get('size'),
        )


def load_json(data):
    data = json.loads(data)
    return Resource.from_dict(data)


def load_json_list(data):
    data = json.loads(data)
    return [
        Resource.from_dict(record)
        for record in data
    ]


def dump_json(resource, indent=None):
    return json.dumps(resource.to_dict(), indent=indent)


def dump_json_list(resources, indent=None):
    return json.dumps(
        [
            resource.to_dict()
            for resource in resources
        ],
        indent=indent
    )


def normalize_namespace(path):
    res = path.strip('/')
    assert res, 'should not be empty'
    assert '//' not in res
    return '/{}/'.format(res)


def normalize_resource_name(resource_name):
    res = resource_name.strip('/')
    assert res, 'should not be empty'
    assert '//' not in res
    return res


def safe_join_path(*parts):
    normalized_parts = []
    first = True
    for part in parts:
        part = part.rstrip('/') if first else part.strip('/')
        if part:
            normalized_parts.append(part)
        first = False
    return os.path.join(*normalized_parts)
