import os
import logging
import tempfile

import path as path_utils


_DESCRIPTION_ = 'description'
_RESOURCE_ = 'resource'


class FlatStorage(object):
    def __init__(self, root_path):
        self._root_path = os.path.abspath(root_path)
        path_utils.ensure_dir(root_path)

    def list(self):
        return os.listdir(self._root_path)

    def exists(self, resource_name):
        return os.path.exists(self._path_to(resource_name))

    def is_fetched(self, resource_name):
        return os.path.exists(self._path_to(resource_name, _DESCRIPTION_))

    def description(self, resource_name):
        assert self.is_fetched(resource_name)
        with open(self._path_to(resource_name, _DESCRIPTION_)) as f:
            return f.read()

    def path_to_resource(self, resource_name):
        assert self.is_fetched(resource_name)
        return self._path_to(resource_name, _RESOURCE_)

    def add(self, resource_name, description):
        assert not self.is_fetched(resource_name)
        path_utils.ensure_dir(self._path_to(resource_name))
        _log.debug('will add %s', resource_name)
        return _AddResource(
            path_to_resource=self._path_to(resource_name, _RESOURCE_),
            path_to_description=self._path_to(resource_name, _DESCRIPTION_),
            description=description,
        )

    def remove(self, resource_name):
        path_utils.remove_dir(self._path_to(resource_name))
        _log.info('removed %s', resource_name)

    def remove_description(self, resource_name):
        if self.is_fetched(resource_name):
            os.unlink(self._path_to(resource_name, _DESCRIPTION_))
            _log.debug('remove description for %s', resource_name)

    def _path_to(self, resource_name, meta_file=''):
        return os.path.join(self._root_path, resource_name, meta_file)


class _AddResource(object):
    def __init__(self, path_to_resource, path_to_description, description):
        self._path_to_resource = path_to_resource
        self._path_to_description = path_to_description
        self._description = description

    def __enter__(self):
        return self._path_to_resource

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            _log.debug('will put description at %s', self._path_to_description)
            with tempfile.NamedTemporaryFile(dir=os.path.dirname(self._path_to_description), delete=False) as fp:
                fp.write(self._description)
                fp.flush()
                os.fsync(fp.fileno())
                fp.close()
                os.chmod(fp.name, 0o644)
                os.rename(fp.name, self._path_to_description)
                dirfd = os.open(os.path.dirname(self._path_to_description), os.O_DIRECTORY)
                try:
                    os.fsync(dirfd)
                finally:
                    os.close(dirfd)
            _log.debug('done put description at %s', self._path_to_description)


_log = logging.getLogger(__name__)
