import collections
import gevent
import gevent.event
import json
import logging
import os
import random
import sys

import infra.callisto.libraries.copier as copier
import infra.callisto.deploy.resource as deploy_resource

import flat_storage
import tree_storage


class Storage(object):
    def __init__(self, root_path, cleanup_cooldown, sleep_time=150):
        self._flat = flat_storage.FlatStorage(os.path.join(root_path, 'flat'))
        self._tree = tree_storage.LinksTree(os.path.join(root_path, 'tree'))
        self._cleanup_cooldown = cleanup_cooldown
        self._sleep_time = sleep_time
        self._current_resource = None
        self._wakeuper = gevent.event.Event()

    @property
    def current_resource(self):
        return self._current_resource

    def list_ready(self):
        ready = []
        for flat_name in self._flat.list():
            if self._flat.is_fetched(flat_name):
                try:
                    ready.append(self._flat_name_to_resource(flat_name))
                except ValueError:
                    pass
        return ready

    def is_resource_ready(self, resource):
        flat_name = _flat_name(resource)
        if self._flat.is_fetched(flat_name):
            stored_resource = self._flat_name_to_resource(_flat_name(resource))
            if stored_resource.rbtorrent:
                return resource.rbtorrent == stored_resource.rbtorrent
            else:
                return True
        return False

    def invalidate(self, resource):
        """Note that invalidate does not remove the resource"""
        self._flat.remove_description(_flat_name(resource))

    def cleanup_except(self, keep_resources):
        keep_flatnames = {_flat_name(resource) for resource in keep_resources}
        for flat_name in self._flat.list():
            if flat_name in keep_flatnames:
                continue

            if not self._flat.is_fetched(flat_name):
                _log.debug('cleanup: removing incomplete fetch [%s]', flat_name)
                self._flat.remove(flat_name)
            else:
                _log.debug('cleanup: removing ready [%s]', flat_name)
                try:
                    resource_ = self._flat_name_to_resource(flat_name)
                except (ValueError, RuntimeError):
                    _log.error('unable to parse description of [{}]'.format(flat_name))
                else:
                    if resource_ not in keep_resources:
                        self._tree.unset_link(resource_)
                        self._flat.remove(flat_name)
            if self._cleanup_cooldown:
                # avoid to flood a disks on mass cleanup
                gevent.sleep(self._cleanup_cooldown)

    def download(self, resource, copier_opts):
        with self._flat.add(_flat_name(resource), deploy_resource.dump_json(resource)) as dst:
            try:
                self._current_resource = resource
                copier.sky_get(
                    resource_url=resource.rbtorrent,
                    dst_dir=dst,
                    **copier_opts
                )
                self._tree.set_link(resource, dst)
            finally:
                self._current_resource = None

    def wakeup(self):
        self._wakeuper.set()

    def sleep(self):
        self._wakeuper.wait(
            timeout=random.uniform(self._sleep_time * 0.5, self._sleep_time * 1.5)
        )
        self._wakeuper.clear()

    def _flat_name_to_resource(self, flat_name):
        return deploy_resource.ResolvedResource.from_dict(json.loads(self._flat.description(flat_name)))


def _flat_name(resource):
    return deploy_resource.safe_join_path(
        resource.namespace,
        '__resources__',
        resource.name,
        '__resource__',
    ).strip('/').replace('/', '_')


class Deployer(object):
    def __init__(self, root_path, tracker_client, cleanup_cooldown, sleep_time=600):
        self._storage = Storage(root_path, cleanup_cooldown)
        self._resolver = ResourceResolver(tracker_client)
        self._targets = Targets()
        self._sleep_time = sleep_time

        self._resolve_greenlet = None
        self._download_greenlet = None

    def set_targets(self, targets):
        self._targets.set_targets(targets)
        self._resolver.wakeup()

    def status(self):
        report = {
            resource: 'idle'
            for resource in self._targets.get_unresolved()
        }

        report.update({
            resolved_resource.get_resource(): self._status(resolved_resource)
            for resolved_resource in self._storage.list_ready()
        })

        report.update({
            resolved_resource.get_resource(): self._status(resolved_resource)
            for resolved_resource in self._targets.get_resolved()
        })

        return [
            {
                'name': res.name,
                'namespace': res.namespace,
                'status': status,
            } for res, status in report.iteritems()
        ]

    def _status(self, resource):
        if self._storage.is_resource_ready(resource):
            return 'prepared'
        if resource == self._storage.current_resource:
            return 'downloading'
        return 'idle'

    def start(self):
        self._resolve_greenlet = _start_greenlet(self._resolve_greenlet, self._resolve_loop)
        self._download_greenlet = _start_greenlet(self._download_greenlet, self._download_loop)

    def _restart_download_loop(self):
        _log.info('restarting download loop')
        if self._download_greenlet is not None:
            self._download_greenlet.kill(timeout=30)
            if not self._download_greenlet.dead:
                _log.fatal('could not stop download loop, exiting')
                sys.exit(1)
        return _start_greenlet(self._download_greenlet, self._download_loop)

    def _cleanup(self):
        try:
            all_targets = set(self._targets.get_unresolved())
            if len(all_targets):
                _log.debug('cleanup')
                self._storage.cleanup_except(all_targets)
            else:
                _log.debug('no targets, skip cleanup')
        except (IOError, RuntimeError) as e:
            _log.exception(e)

    def _download(self, resource):
        _log.debug('will download %s', resource)
        copier_opts = self._targets.get_copier_args(resource)
        self._storage.download(resource, copier_opts)
        _log.info('downloaded %s', resource)

    def _download_loop(self):
        _log.debug('start download loop')
        while True:
            try:
                self._cleanup()
                for resource in self._targets.get_resolved():
                    try:
                        if self._storage.is_resource_ready(resource):
                            continue
                        self._storage.invalidate(resource)
                        self._download(resource)
                    except (IOError, OSError, RuntimeError) as exc:
                        _log.exception(exc)
            finally:
                self._storage.sleep()

    def _resolve_all(self):
        resolved = self._resolver.resolve(
            self._targets.get_unresolved()
        )

        if resolved:
            self._targets.set_resolved(resolved)

    def _stop_obsolete_fetching(self):
        if self._storage.current_resource not in self._targets.get_resolved():
            _log.debug(
                'Resource %s have been changed or deleted from targets',
                self._storage.current_resource
            )
            self._download_greenlet = self._restart_download_loop()

    def _wakeup_storage(self):
        self._storage.wakeup()

    def _resolve_loop(self):
        _log.debug('start resolve loop')
        while True:
            try:
                self._resolve_all()
                if self._storage.current_resource:
                    self._stop_obsolete_fetching()
                else:
                    self._wakeup_storage()
            except Exception as exc:
                _log.exception(exc)
            finally:
                self._resolver.sleep()


class Targets(object):
    def __init__(self):
        self._unresolved_resources = dict()
        self._resolved_resources = set()

    def set_targets(self, targets):
        _log.info('setting targets (was %s new %s)', len(self._unresolved_resources), len(targets))
        self._unresolved_resources = targets

    def set_resolved(self, resources):
        resolved = set()

        for res in resources:
            if res.get_resource() in self._unresolved_resources:
                resolved.add(res)

        self._resolved_resources = resolved

    def get_unresolved(self):
        return self._unresolved_resources.keys()

    def get_copier_args(self, resource):
        return self._unresolved_resources[resource.get_resource()]

    def get_resolved(self):
        for resolved_res in sorted(self._resolved_resources, key=_sort_resources_key):
            if resolved_res.get_resource() in self._unresolved_resources:
                yield resolved_res


class ResourceResolver(object):
    def __init__(self, tracker_client, sleep_time=600):
        self._tracker_client = tracker_client
        self._wakeuper = gevent.event.Event()
        self._sleep_time = sleep_time

    def resolve(self, resources):
        names_by_ns = collections.defaultdict(set)
        for resource in resources:
            names_by_ns[resource.namespace].add(resource.name)

        resolved_resources = set()
        resolved_resources.update(*(
            self._tracker_client.resolve_many(namespace, names)
            for namespace, names in names_by_ns.iteritems()
        ))

        _log.info('Resolved %s / %s resources', len(resolved_resources), len(resources))

        return resolved_resources

    def wakeup(self):
        self._wakeuper.set()

    def sleep(self):
        self._wakeuper.wait(
            timeout=random.uniform(self._sleep_time * 0.5, self._sleep_time * 1.5)
        )
        self._wakeuper.clear()


def namespace_path(root_path, namespace):
    return tree_storage.namespace_path(os.path.join(root_path, 'tree'), namespace)


def resource_path(root_path, namespace, name):
    return tree_storage.resource_path(os.path.join(root_path, 'tree'), deploy_resource.Resource(namespace, name))


def _sort_resources_key(resource):
    # from old to new,
    # any WebTier0 related at the last,
    # then gemini,
    # multi_level_cache_replicas before gemini,
    # remote_storages before multi_level_cache_replicas,
    # then in some stable order
    return (
        'WebTier0' in resource.name,
        'GeminiTier' in resource.name,
        resource.namespace,
        'multi_level_cache_replicas' in resource.name,
        'remote_storage' in resource.name,
        hash(resource.name)
    )


def _start_greenlet(greenlet, runner):
    return gevent.spawn(runner) if greenlet is None or greenlet.dead else greenlet


_log = logging.getLogger(__name__)
