import collections
import os
import logging

import infra.callisto.deploy.resource as deploy_resource

import path as path_utils


NamingScheme = collections.namedtuple('NamingScheme', ['resources', 'resource'])
_DEFAULT_NAMING_SCHEME = NamingScheme('@resources', '@resource')


class LinksTree(object):
    def __init__(self, root_path, naming_scheme=None):
        self._root_path = os.path.abspath(root_path)
        path_utils.ensure_dir(root_path)
        self._naming_scheme = naming_scheme or _DEFAULT_NAMING_SCHEME

    def set_link(self, resource_, path_to_resource):
        assert resource_ and path_to_resource
        path = self.resource_path(resource_)
        path_utils.ensure_dir(os.path.dirname(path))
        path_utils.set_symlink(path_to_resource, path)

    def unset_link(self, resource_):
        path = self.resource_path(resource_)
        path_utils.unset_symlink(path)
        parent_dir = os.path.dirname(path)
        if os.path.exists(parent_dir) and not os.listdir(parent_dir):
            path_utils.remove_dir(parent_dir)

    def namespace_path(self, namespace):
        return namespace_path(self._root_path, namespace, self._naming_scheme.resources)

    def resource_path(self, resource):
        return deploy_resource.safe_join_path(
            self.namespace_path(resource.namespace),
            resource.name,
            self._naming_scheme.resource
        )


def namespace_path(root_path, namespace, delimiter=_DEFAULT_NAMING_SCHEME.resources):
    return deploy_resource.safe_join_path(
        root_path,
        namespace,
        delimiter,
    )


def resource_path(root_path, resource_, delimiter=_DEFAULT_NAMING_SCHEME.resource):
    return deploy_resource.safe_join_path(
        namespace_path(root_path, resource_.namespace),
        resource_.name,
        delimiter,
    )


_log = logging.getLogger(__name__)
