import collections
import flask
import json

import infra.callisto.deploy.resource as deploy_resource
import infra.callisto.deploy.tracker.core.pool as pool
import infra.callisto.deploy.tracker.core.table as table
import infra.callisto.deploy.tracker.core.tracker as tracker


YASM_DEFAULT_SIG_SUFFIX = 'dmmv'


class Counter(object):
    def __init__(self):
        self._counters = collections.defaultdict(int)

    def inc(self, name, cnt):
        self._counters[name] += cnt

    def solomon(self):
        return {
            'sensors': [
                {
                    'labels': dict(sensor=key),
                    'value': value,
                    'kind': 'RATE',
                }
                for key, value in self._counters.items()
            ]
        }

    def yasm(self):
        return [
            ['{}_{}'.format(key, YASM_DEFAULT_SIG_SUFFIX), value]
            for key, value in self._counters.items()
        ]


class Context(object):
    def __init__(self, config):
        self.tracker = tracker.Tracker(config)
        self.counter = Counter()


def create(config):
    app = flask.Flask(__name__)
    app.context = Context(config)
    app.add_url_rule('/<path:full_resource_id>', view_func=_api_resolve_full, methods=['GET'])
    app.add_url_rule('/_solomon', view_func=_api_solomon, methods=['GET'])
    app.add_url_rule('/_unistat', view_func=_api_yasm_unistat, methods=['GET'])

    app.register_blueprint(v1_bp(), url_prefix='/v1')
    app.register_blueprint(v2_bp(), url_prefix='/v2')
    return app


def v1_bp():
    bp = flask.Blueprint('v1', __name__)
    bp.add_url_rule('/<path:namespace>', view_func=_api_register_resources, methods=['POST'])
    bp.add_url_rule('/<path:namespace>', view_func=_api_resolve, methods=['GET'])
    return bp


def v2_bp():
    bp = flask.Blueprint('v2', __name__)
    bp.add_url_rule('/register/<path:namespace>', view_func=_api_register_resources, methods=['POST'])
    bp.add_url_rule('/resolve/<path:namespace>', view_func=_api_resolve, methods=['GET'])
    bp.add_url_rule('/resolve_many/<path:namespace>', view_func=_api_resolve_many, methods=['POST'])
    bp.add_url_rule('/match/<path:namespace>', view_func=_api_match_resources, methods=['GET'])
    return bp


def _api_register_resources(namespace):
    resources = [
        deploy_resource.ResolvedResource(
            namespace=namespace,
            name=record['name'],
            rbtorrent=record['rbtorrent'],
            size=record.get('size'),
        )
        for record in json.loads(flask.request.json)
    ]
    try:
        _tracker().register_many(namespace, resources)
        _counter().inc('register_ok_resources', len(resources))
        _counter().inc('register_ok_requests', 1)
    except table.RbtorrentDiffers as exc:
        _counter().inc('register_conflict_requests', 1)
        return flask.Response('{} rbtorrent differs'.format(exc.resource_name), status=400)
    except pool.NoResourcesLeft:
        _counter().inc('register_no_clients_left', 1)
        return flask.Response('no register clients left, try again later', 503)
    except Exception:
        _counter().inc('register_failure_requests', 1)
        raise
    return flask.Response('OK', status=200)


def _api_resolve(namespace):
    if 'name' not in flask.request.args:
        return flask.Response('pass `name` parameter', 400)
    name = flask.request.args['name']
    try:
        resource_ = _resolve(namespace, name)
    except pool.NoResourcesLeft:
        _counter().inc('resolve_no_clients_left', 1)
        return flask.Response('no resolve clients left, try again later', 503)

    if resource_:
        _counter().inc('resolve_ok_requests', 1)
        return flask.Response(deploy_resource.dump_json(resource_, indent=4), 200, mimetype='application/json')
    else:
        _counter().inc('resolve_not_found_requests', 1)
        return flask.Response('resource not found', 404)


def _api_resolve_many(namespace):
    if 'names' not in flask.request.json:
        return flask.Response('pass `names` parameter', 400)
    names = frozenset(flask.request.json['names'])
    try:
        resources_ = _resolve_many(namespace, names)
    except pool.NoResourcesLeft:
        _counter().inc('resolve_no_clients_left', 1)
        return flask.Response('no resolve clients left, try again later', 503)
    _counter().inc('resolve_many_ok_requests', 1)
    return flask.Response(deploy_resource.dump_json_list(resources_, indent=4), 200, mimetype='application/json')


def _api_match_resources(namespace):
    if 'name_regexp' not in flask.request.args:
        return flask.Response('pass `name_regexp` parameter', 400)
    name_regexp = flask.request.args['name_regexp']
    resources = _tracker().list_resources(namespace=namespace, name_regexp=name_regexp)
    _counter().inc('match_requests', 1)
    return deploy_resource.dump_json_list(resources)


def _api_resolve_full(full_resource_id):
    return flask.Response(json.dumps(
        _tracker().resolve_resource_id(full_resource_id),
        indent=4,
    ), mimetype='application/json')


def _api_solomon():
    return flask.Response(json.dumps(
        _counter().solomon(),
        indent=4,
    ), mimetype='application/json')


def _api_yasm_unistat():
    return flask.Response(json.dumps(
        _counter().yasm(),
        indent=4,
    ), mimetype='application/json')


def _resolve(namespace, name):
    result = _tracker().resolve_one(namespace, name)
    if result:
        _counter().inc('resolve_yt_ok_requests', 1)
    else:
        _counter().inc('resolve_yt_not_found_requests', 1)
    return result


def _resolve_many(namespace, names):
    result = _tracker().resolve_many(namespace, names)
    if result:
        _counter().inc('resolve_many_yt_ok_requests', 1)
    else:
        _counter().inc('resolve_yt_not_found_requests', 1)
    return result


def _tracker():
    return flask.current_app.context.tracker


def _counter():
    return flask.current_app.context.counter
