import gevent.monkey
gevent.monkey.patch_all()  # noqa
import gevent.pywsgi as pywsgi

import os
import logging
import logging.handlers
import argparse

import google.protobuf.text_format
import infra.callisto.deploy.tracker.proto.config_pb2 as config_pb2  # noqa

import app


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)


def configure_yt_logging(enable, root, debug, formatter):
    import yt.logger
    yt.logger.LOGGER.setLevel(logging.DEBUG if debug else logging.INFO)
    for handler in yt.logger.LOGGER.handlers[:]:
        yt.logger.LOGGER.removeHandler(handler)
    if enable:
        handler = logging.handlers.RotatingFileHandler(os.path.join(root, 'yt.log'), maxBytes=1024 ** 3, backupCount=5)
        handler.setFormatter(formatter)
        yt.logger.LOGGER.addHandler(handler)


def configure_logging(yt, debug, console, http):
    root = './logs'

    formatter = logging.Formatter('%(asctime)s [%(levelname).1s] [%(name)s] %(message)s')

    info_handler = logging.handlers.RotatingFileHandler(os.path.join(root, 'info.log'), maxBytes=1024 ** 3, backupCount=5)
    info_handler.setLevel(logging.INFO)
    info_handler.setFormatter(formatter)

    logging.getLogger().setLevel(level=logging.DEBUG if debug else logging.INFO)
    logging.getLogger().addHandler(info_handler)

    if debug:
        debug_handler = logging.handlers.RotatingFileHandler(os.path.join(root, 'debug.log'), maxBytes=1024 ** 3, backupCount=5)
        debug_handler.setLevel(logging.DEBUG)
        debug_handler.setFormatter(formatter)
        logging.getLogger().addHandler(debug_handler)

    if console:
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.DEBUG if debug else logging.INFO)
        console_handler.setFormatter(formatter)
        logging.getLogger().addHandler(console_handler)

    if http:
        http_handler = logging.handlers.RotatingFileHandler(os.path.join(root, 'http.log'), maxBytes=1024 ** 3, backupCount=5)
        http_handler.setLevel(logging.DEBUG if debug else logging.INFO)
        http_handler.setFormatter(formatter)
        logging.getLogger('http').addHandler(http_handler)
        logging.getLogger('http').propagate = False

    logging.getLogger('werkzeug').setLevel(logging.ERROR)
    configure_yt_logging(yt, root, debug, formatter)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--proxy', help='Yt proxy', type=str)
    parser.add_argument('--port', help='Server port', type=int)
    parser.add_argument('--console', help='Dump logs to stdout', action='store_true', default=False)
    parser.add_argument('--readonly', help='Readonly mode', action='store_true', default=True)
    parser.add_argument('--not-readonly', help='Not readonly mode', action='store_false', dest='readonly')
    parser.add_argument('--config', help='Config file')
    return parser.parse_args()


def parse_config(args):
    config = config_pb2.TServerConfig()
    if args.config:
        with open(args.config) as config_file:
            google.protobuf.text_format.Parse(config_file.read(), config)

    # override with command line options and defaults
    config.Readonly = args.readonly
    if args.port:
        config.Port = args.port
    if args.proxy:
        config.Yt.Proxy = args.proxy
    if not config.Cache.RecordCountLimit:
        config.Cache.RecordCountLimit = 1000
    if not config.Cache.RecordTtl:
        config.Cache.RecordTtl = 30

    return config


def main():
    args = parse_args()
    config = parse_config(args)
    configure_logging(yt=config.Logging.Yt, debug=config.Logging.Debug, http=config.Logging.Http, console=args.console)
    logging.info('Start parameters:\n%s', config)
    logging.info('Open port %s', config.Port)
    pywsgi.WSGIServer(('::', config.Port), app.create(config),
                      log=logging.getLogger('http')).serve_forever()
