import logging
import requests

import infra.callisto.deploy.resource as deploy_resource


_RESOLVE_TIMEOUT = 20
_REGISTER_TIMEOUT = 30


class RbTorrentDiffers(ValueError):
    pass


class ResolveError(RuntimeError):
    pass


class ResourceNotFound(ResolveError):
    pass


def _join_url(url, *parts):
    return deploy_resource.safe_join_path(url, *parts)


def _register_resources(tracker_url, namespace, resources):
    url = _join_url(tracker_url, 'v1', deploy_resource.normalize_namespace(namespace))
    resp = requests.post(
        url=url,
        json=deploy_resource.dump_json_list(resources),
        timeout=_REGISTER_TIMEOUT,
    )
    if resp.status_code == 400 and 'rbtorrent differs' in resp.content:
        raise RbTorrentDiffers(resp.content)
    resp.raise_for_status()


def _resolve_resource(tracker_url, namespace, name):
    url = _join_url(tracker_url, 'v1', deploy_resource.normalize_namespace(namespace))
    resp = requests.get(
        url=url,
        timeout=_RESOLVE_TIMEOUT,
        params={'name': name},
    )
    if resp.status_code == 404:
        raise ResourceNotFound('{}{} not found'.format(namespace, name))
    resp.raise_for_status()
    return deploy_resource.ResolvedResource.from_dict(resp.json())


def _resolve_resources(tracker_url, namespace, names):
    url = _join_url(tracker_url, 'v2/resolve_many', deploy_resource.normalize_namespace(namespace))
    resp = requests.post(
        url=url,
        timeout=_RESOLVE_TIMEOUT,
        json={'names': list(names)},
        headers={'Content-type': 'application/json'},
    )
    if resp.status_code == 404:
        raise ResourceNotFound('{} not found in namespace {}'.format(names, namespace))
    resp.raise_for_status()
    return [deploy_resource.ResolvedResource.from_dict(resource) for resource in resp.json()]


def _match_resources(tracker_url, namespace, name_regexp):
    url = _join_url(tracker_url, 'v2/match', deploy_resource.normalize_namespace(namespace))
    resp = requests.get(
        url=url,
        timeout=_RESOLVE_TIMEOUT,
        params={'name_regexp': name_regexp},
    )
    resp.raise_for_status()
    return [deploy_resource.ResolvedResource.from_dict(resource) for resource in resp.json()]


class _RegisterRequest(object):
    def __init__(self, tracker_url, namespace):
        self._tracker_url = tracker_url
        self._namespace = namespace
        self._resources = []
        self._committed = False

    def add_resource(self, name, rbtorrent, size=None):
        self._resources.append(
            deploy_resource.ResolvedResource(
                namespace=self._namespace,
                name=name,
                rbtorrent=rbtorrent,
                size=size
            )
        )

    def register(self):
        assert self._resources
        assert not self._committed

        logging.info(
            'will register %s resources at %s in %s',
            len(self._resources),
            self._namespace,
            self._tracker_url,
        )
        _register_resources(self._tracker_url, self._namespace, self._resources)
        self._committed = True
        logging.info(
            'registered %s resources at %s in %s',
            len(self._resources),
            self._namespace,
            self._tracker_url,
        )


class Client(object):
    def __init__(self, url):
        self._url = url

    def make_register_request(self, namespace):
        return _RegisterRequest(self._url, namespace)

    def resolve_one(self, namespace, name):
        return _resolve_resource(self._url, namespace, name)

    def resolve_many(self, namespace, names):
        return _resolve_resources(self._url, namespace, names)

    def match_resources(self, namespace, name_regexp):
        return _match_resources(self._url, namespace, name_regexp)
