import os

import retry
import infra.callisto.libraries.copier as copier
import infra.callisto.deploy.resource as deploy_resource


def _parents(path):
    assert not path.startswith('/') and not path.endswith('/'), path
    while path:
        yield path
        path = os.path.dirname(path)


def _filter_leaves(resources):
    resources = sorted(resources, key=lambda x: (x.name.count('/'), x.name), reverse=True)
    seen_parents = set()
    result = []
    for resource in resources:
        if resource.name not in seen_parents:
            result.append(resource)
        seen_parents.update(_parents(resource.name))
    return result


def _cut_root_dirs(resources):
    def cut_root_dir(name):
        parts = name.strip('/').split('/')
        return '/'.join(parts[1:])

    return [
        deploy_resource.ResolvedResource(namespace=x.namespace, name=cut_root_dir(x.name), size=x.size, rbtorrent=x.rbtorrent)
        for x in resources
    ]


def _common_root_dir(resources):
    first_dirs = {x.name.split('/')[0] for x in resources}
    if len(first_dirs) > 1:
        raise ValueError('various root names: %s', sorted(first_dirs))
    return first_dirs.pop() if first_dirs else None


def download_recursive(dst_path, resources, resource_attempts_count=1, **copier_opts):
    @retry.retry(tries=resource_attempts_count, delay=30)
    def _sky_get(rbtorrent_, path_, **copier_args_):
        copier.sky_get(rbtorrent_, path_, **copier_args_)

    _common_root_dir(resources)
    resources = _filter_leaves(resources)
    resources = _cut_root_dirs(resources)

    for resource in resources:
        full_path = os.path.join(dst_path, resource.name)
        if not os.path.exists(full_path):
            os.makedirs(full_path)
        _sky_get(resource.rbtorrent, full_path, **copier_opts)
