import os
import collections


_FileSpec = collections.namedtuple('_FileSpec', ['path', 'size'])
_DirSpec = collections.namedtuple('_DirSpec', ['path', 'files', 'dirs', 'size'])


def inspect_directory(directory):
    return _dir_spec(directory, directory)


def relative_path(root_dir_path, sub_dir_path):
    if not sub_dir_path.startswith(root_dir_path):
        raise ValueError()
    rel_path = os.path.relpath(sub_dir_path, root_dir_path)
    return os.path.normpath(os.path.join(os.path.basename(root_dir_path), rel_path))


def list_dir(path, dirs=True, files=True):
    result = []
    for sub_path in os.listdir(path):
        full_path = os.path.join(path, sub_path)
        if not os.path.islink(full_path):
            if files and os.path.isfile(full_path):
                result.append(full_path)
            if dirs and os.path.isdir(full_path):
                result.append(full_path)
    return result


def _dir_spec(root_path, current_directory):
    files = set()
    sub_dirs = set()

    for sub_dir_full_path in list_dir(current_directory):
        if os.path.isfile(sub_dir_full_path):
            files.add(_file_spec(root_path, sub_dir_full_path))
        else:
            sub_dirs.add(_dir_spec(root_path, sub_dir_full_path))

    return _DirSpec(
        path=relative_path(root_path, current_directory),
        files=frozenset(files),
        dirs=frozenset(sub_dirs),
        size=sum(x.size for x in files | sub_dirs),
    )


def _file_spec(root_path, path):
    size = os.stat(path).st_size
    return _FileSpec(
        path=relative_path(root_path, path),
        size=size,
    )
