import os
import pytest

import infra.callisto.deploy.tracker.client.inspect as inspect


def test_relative_path():
    assert inspect.relative_path('/a/b/c', '/a/b/c/d/e') == 'c/d/e'
    assert inspect.relative_path('/a/b/c', '/a/b/c') == 'c'
    assert inspect.relative_path(os.getcwd() + '/a', os.getcwd() + '/a/b') == 'a/b'
    with pytest.raises(ValueError):
        inspect.relative_path('/a/b/c', '/a/b')
    with pytest.raises(ValueError):
        inspect.relative_path('/a/b/c', '/a/b/d')


def test_inspect_directory():
    test_path = os.path.join(os.getcwd(), 'test_inspect_directory')
    os.makedirs(test_path)

    path_to_a = os.path.join(test_path, 'a')
    os.makedirs(path_to_a)
    path_to_x_file = os.path.join(path_to_a, 'x')
    with open(path_to_x_file, 'w') as f:
        f.write('x' * 256)
    os.symlink(path_to_x_file, os.path.join(test_path, 'a_link'))

    specs = inspect.inspect_directory(test_path)

    assert specs.size == 256
    assert not specs.files, 'there are no files'
    assert specs.dirs, 'there is `a` dir'
    assert not list(specs.dirs)[0].dirs, 'there are no dirs'
    assert list(specs.dirs)[0].files, 'there is `x` file'
    assert list(specs.dirs)[0].size == 256, '`x` file size'
    assert list(list(specs.dirs)[0].files)[0].size == 256

    assert specs == inspect.inspect_directory(test_path), 'specs should have not change'
    os.unlink(os.path.join(test_path, 'a_link'))
    assert specs == inspect.inspect_directory(test_path), 'symlinks are ignored'

    with open(os.path.join(test_path, 'b'), 'w') as f:
        f.write('x')

    assert specs != inspect.inspect_directory(test_path), 'new file appeared'
    os.remove(os.path.join(test_path, 'b'))
    assert specs == inspect.inspect_directory(test_path), 'new file removed'
