import collections
import datetime
import itertools
import logging
import threading

_ResourceCacheItem = collections.namedtuple('_ResourceCacheItem', ['value', 'end_of_life'])


class ResourceCache(object):
    def __init__(self, ttl, records_limit):
        self._cache = collections.OrderedDict()
        self._ttl = datetime.timedelta(seconds=ttl)
        self._cache_size_limit = records_limit
        self._truncation_lock = threading.Lock()

    def remove_cached(self, namespace, names):
        for name in names:
            key = (namespace, name)
            if key in self._cache:
                del self._cache[key]

    def get_cached(self, namespace, names):
        now = datetime.datetime.now()
        found = {}

        for name in names:
            item = self._cache.get((namespace, name))
            if item:
                if item.end_of_life < now:
                    del self._cache[(namespace, name)]
                else:
                    found[name] = item.value

        return found

    def set_cached(self, namespace, resources, nones=None):
        nones = nones or []
        now = datetime.datetime.now()

        if len(self._cache) >= self._cache_size_limit:
            self._truncate()

        ttl = now + self._ttl
        for resource in resources:
            self._cache[(namespace, resource.name)] = _ResourceCacheItem(resource, ttl)

        for name in nones:
            self._cache[(namespace, name)] = _ResourceCacheItem(None, ttl)

    def _truncate(self):
        if self._truncation_lock.acquire(blocking=False):
            try:
                for key in itertools.islice(self._cache, 0, self._cache_size_limit / 4):
                    del self._cache[key]
            finally:
                self._truncation_lock.release()
            _log.debug('truncated, new size: %s', len(self._cache))
        else:
            _log.debug('truncation bypassed')


_log = logging.getLogger(__name__)
