import datetime
import logging
import os.path

import yt.yson as yson

import infra.callisto.deploy.resource as deploy_resource
import infra.callisto.libraries.yt as yt_utils

YT_ROOT = '//home/cajuper/tracker'


class RbtorrentDiffers(RuntimeError):
    def __init__(self, resource_name):
        self.resource_name = resource_name


class NamespaceTable(yt_utils.SortedYtTable):
    schema = yson.to_yson_type([
        {'name': 'name',        'type': 'string', 'sort_order': 'ascending', 'required': True},
        {'name': 'rbtorrent',   'type': 'string'},
        {'name': 'time',        'type': 'string'},
        {'name': 'size',        'type': 'int64'},
    ], attributes={"strict": True, "unique_keys": True})

    tablet_cell_bundle = 'cajuper'

    def __init__(self, namespace, yt_client, readonly=True):
        super(NamespaceTable, self).__init__(
            yt_client=yt_client,
            path=namespace_to_path(namespace),
            readonly=readonly,
        )
        self._logger = logging.getLogger(namespace)
        self._namespace = namespace

    def _on_init_hook(self):
        pass

    def resolve_one(self, name):
        results = self.resolve_many([name])
        results_number = len(results)

        if results_number > 1:
            self._logger.error('Have got %s results for %s', results_number, name)

        return results.pop() if results_number == 1 else None

    def resolve_many(self, names):
        return {
            deploy_resource.ResolvedResource(
                namespace=self._namespace,
                name=row['name'],
                rbtorrent=row['rbtorrent'],
                size=row.get('size'),
            )
            for row in self._lookup_rows([{'name': name} for name in names])
        }

    def register_many(self, resources):
        resources = {res.name: res for res in resources}
        with self._transaction() as transaction:
            transaction_id = transaction.transaction_id
            self._logger.info('starting transaction %s', transaction_id)

            rows = self._lookup_rows([{'name': name} for name in resources])
            existing_resources = {row['name']: row['rbtorrent'] for row in rows}
            new_resources = sorted(set(resources) - set(existing_resources))

            for name in set(resources) & set(existing_resources):
                if resources[name].rbtorrent != existing_resources[name]:
                    logging.warning('resource %s has different rbtorrents', name)
                    raise RbtorrentDiffers(name)
                else:
                    self._logger.info('skip %s', name)

            rows = [{
                'name': name,
                'rbtorrent': resources[name].rbtorrent,
                'size': resources[name].size,
                'time': datetime.datetime.now().isoformat(),
            } for name in new_resources]
            self._insert_rows(rows)
        if len(new_resources) > 100:
            some_registered = ' '.join(new_resources[:10]) + ' ... ' + ' '.join(new_resources[-10:])
        else:
            some_registered = ' '.join(new_resources)
        self._logger.info(
            'done transaction %s. registered %s resources: [%s]',
            transaction_id, len(new_resources), some_registered,
        )

    def delete_many(self, names_of_resources):
        with self._transaction() as transaction:
            transaction_id = transaction.transaction_id
            self._logger.info('starting transaction %s', transaction_id)
            to_delete = [{'name': name} for name in names_of_resources]
            self._yt_client.delete_rows(self._path, to_delete, format='json')
        self._logger.info('done transaction %s. deleted %s', transaction_id, len(names_of_resources))

    def list_resources(self, name_regexp=None):
        if name_regexp is None:
            request = '* from [{}]'.format(self._path)
        else:
            request = '* from [{}] where regex_partial_match("{}", name)'.format(self._path, name_regexp)
        return [
            deploy_resource.ResolvedResource(
                namespace=self._namespace,
                name=row['name'],
                rbtorrent=row['rbtorrent'],
                size=row.get('size'),
            )
            for row in self._select_rows(request)
        ]


def namespace_to_path(namespace):
    return os.path.join(YT_ROOT, namespace.lstrip('/'), 'table')


def path_to_namespace(path):
    return deploy_resource.normalize_namespace(
        os.path.relpath(path, YT_ROOT).replace('/table', '')
    )
