import gevent
import logging

import infra.callisto.deploy.resource as deploy_resource
import infra.callisto.libraries.yt as yt_utils

import cache
import pool
import table
import utils


class _NamespaceCache(gevent.Greenlet):
    def __init__(self, proxy):
        super(_NamespaceCache, self).__init__()
        self._yt_client = yt_utils.create_yt_client(proxy=proxy, use_rpc=True)
        self._cache = set()
        yt_utils.ensure_node(self._yt_client, table.YT_ROOT)

    def _run(self):
        while True:
            try:
                self._update()
            except Exception as exc:
                logging.exception(exc)
            finally:
                gevent.sleep(10)

    def _update(self):
        paths = self._yt_client.search(table.YT_ROOT, node_type='table')
        self._cache = {table.path_to_namespace(path) for path in paths}

    def __contains__(self, item):
        return item in self._cache

    def __iter__(self):
        for namespace in sorted(self._cache):
            yield namespace

    def possible_namespaces(self, resource_id):
        for namespace in utils.possible_namespaces(resource_id):
            if namespace in self._cache:
                yield namespace


class Tracker(object):
    def __init__(self, config):
        self._readonly = config.Readonly
        self._namespaces = _NamespaceCache(config.Yt.Proxy)
        self._resource_cache = cache.ResourceCache(config.Cache.RecordTtl, config.Cache.RecordCountLimit)
        self._clients_pool = pool.Pool([
            yt_utils.create_yt_client(proxy=config.Yt.Proxy, use_rpc=True)
            for _ in range(config.ClientCount)
        ])
        self._read_clients_pool = pool.Pool([
            yt_utils.create_yt_client(proxy=config.Yt.Proxy, use_rpc=True)
            for _ in range(config.ClientCount)
        ])
        self._namespaces.start()

    def register_many(self, namespace, resources):
        namespace = deploy_resource.normalize_namespace(namespace)
        with self._clients_pool.get() as client:
            table_ = table.NamespaceTable(namespace=namespace, yt_client=client, readonly=self._readonly)
            if namespace not in self._namespaces:
                table_.ensure_table()
            table_.ensure_mounted()
            table_.register_many(resources)
            self._resource_cache.set_cached(namespace, resources)

    def resolve_one(self, namespace, name):
        resolved_resources = self.resolve_many(namespace, [name])

        if resolved_resources:
            if len(resolved_resources) == 1:
                return resolved_resources.pop()
            logging.warning('Lookup %s in %s returns %s results', name, namespace, len(resolved_resources))

        return None

    def resolve_many(self, namespace, names):
        namespace = deploy_resource.normalize_namespace(namespace)
        names = {deploy_resource.normalize_resource_name(name) for name in names}

        if namespace in self._namespaces:
            return self._resolve_many(namespace, names)

        return set()

    def _resolve_many(self, namespace, names):
        cached = self._resource_cache.get_cached(namespace, names)
        uncached = names - {name for name in cached}

        if uncached:
            with self._read_clients_pool.get() as client:
                table_ = table.NamespaceTable(namespace=namespace, yt_client=client, readonly=True)
                resolved = table_.resolve_many(uncached)
                self._resource_cache.set_cached(namespace, resolved,
                                                nones=uncached - {resource.name for resource in resolved})
        else:
            resolved = set()

        _log.debug('total %s, cached %s, resolved %s', len(names), len(cached), len(resolved))
        for resource in cached.values():
            if resource:
                resolved.add(resource)
        return resolved

    def resolve_resource_id(self, full_resource_id):
        res = []
        full_resource_id = deploy_resource.normalize_namespace(full_resource_id)
        for namespace in self._namespaces.possible_namespaces(full_resource_id):
            name = utils.separate_resource_id(full_resource_id, namespace)
            with self._read_clients_pool.get() as client:
                table_ = table.NamespaceTable(namespace=namespace, yt_client=client, readonly=True)
                resource = table_.resolve_one(name)
                if resource:
                    res.append({
                        'namespace': namespace,
                        'name': name,
                        'rbtorrent': resource.rbtorrent,
                        'size': resource.size,
                    })
        return res

    def delete_resources(self, namespace, resource_names):
        namespace = deploy_resource.normalize_namespace(namespace)
        assert namespace in self._namespaces
        with self._clients_pool.get() as client:
            table_ = table.NamespaceTable(namespace=namespace, yt_client=client, readonly=self._readonly)
            table_.delete_many(resource_names)
            self._resource_cache.remove_cached(namespace, resource_names)

    def list_resources(self, namespace, name_regexp=None):
        namespace = deploy_resource.normalize_namespace(namespace)
        assert namespace in self._namespaces
        with self._read_clients_pool.get() as client:
            table_ = table.NamespaceTable(namespace=namespace, yt_client=client, readonly=self._readonly)
            return table_.list_resources(name_regexp=name_regexp)


_log = logging.getLogger(__name__)
