import gevent.monkey
gevent.monkey.patch_all()  # noqa

import os
import logging

import infra.callisto.deploy.tracker.core.table as tracker_table
import infra.callisto.deploy.tracker.proto.config_pb2 as config_pb2  # noqa
import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.utils.entities as entities
import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.controllers.utils.yp_utils as yp_utils
import infra.callisto.controllers.utils.bindings as bindings
import infra.callisto.controllers.core.conveyor as conveyor
import infra.callisto.controllers.sdk.registry as registry
import infra.callisto.controllers.user as user
import infra.callisto.controllers.core.loop as loop
import infra.callisto.libraries.yt as yt_utils
import infra.callisto.reports as reports
import infra.callisto.configs as configs
import infra.callisto.deploy.tracker.core.tracker as tracker

import IPython
import traitlets.config.loader


class ReportsKeepers(object):
    """All reports keepers (man, vla, sas, all)"""

    def __init__(self):
        backends = registry.ReportsBackends.Default
        self.sas = conveyor.ReportsKeeper([reports.client.Client(backends.sas)])

    class v2(object):
        _backends = registry.ReportsBackends.V2
        man = conveyor.ReportsKeeper([reports.client.Client(_backends.man)])
        vla = conveyor.ReportsKeeper([reports.client.Client(_backends.vla)])
        sas = conveyor.ReportsKeeper([reports.client.Client(_backends.sas)])

        all = conveyor.ReportsKeeper([
            reports.client.Client(backend_url)
            for backend_url in _backends.all
        ])

    def __repr__(self):
        return self.__class__.__name__


class ConfigsStorages(object):
    """Configs clients (ro man, vla, sas + rw override)"""

    def __init__(self):
        self.override = configs.client.OverrideConfigsClient()

        self.man = configs.client.Client('http://man-configs-cajuper.n.yandex-team.ru')
        self.vla = configs.client.Client('http://vla-configs-cajuper.n.yandex-team.ru')
        self.sas = configs.client.Client('http://sas-configs-cajuper.n.yandex-team.ru')
        self.man_web = configs.client.Client('http://man-configs-web-cajuper.n.yandex-team.ru')
        self.vla_web = configs.client.Client('http://vla-configs-web-cajuper.n.yandex-team.ru')
        self.sas_web = configs.client.Client('http://sas-configs-web-cajuper.n.yandex-team.ru')

    def __repr__(self):
        return self.__class__.__name__


class Controllers(object):
    """Known controllers"""

    def __init__(self):
        class Web(object):
            _ctrls = 'http://ctrl.clusterstate.yandex-team.ru'
            pip = bindings.SlotsController(_ctrls + '/web/pip')
            man = bindings.SlotsController(_ctrls + '/web/prod-man')
            sas = bindings.SlotsController(_ctrls + '/web/prod-sas')
            vla = bindings.SlotsController(_ctrls + '/web/prod-vla')
            main = bindings.SlotsController(_ctrls + '/web/prod')

        class Callisto(object):
            _url = 'http://ctrl.clusterstate.yandex-team.ru/callisto/{location}/'
            man = bindings.SlotsController(_url.format(location='man'))
            sas = bindings.SlotsController(_url.format(location='sas'))
            vla = bindings.SlotsController(_url.format(location='vla'))

        class Video(object):
            _ctrls = 'http://ctrl.clusterstate.yandex-team.ru'
            pip = bindings.SlotsController(_ctrls + '/video/pip')
            man = bindings.SlotsController(_ctrls + '/video/prod-man')
            sas = bindings.SlotsController(_ctrls + '/video/prod-sas')
            vla = bindings.SlotsController(_ctrls + '/video/prod-vla')

        self.web = Web()
        self.callisto = Callisto()
        self.video = Video()


class RootCtrls(object):
    """Root controllers registry"""

    def __init__(self, dct):
        self.__dict__ = dct

    def __repr__(self):
        return self.__class__.__name__


class FastBuilders(object):
    """Fast build shards helper"""
    gencfg_group = gencfg_api.GencfgGroup(
        'VLA_WEB_TIER0_BUILD_EXTRA',
        'stable-156-r1054',
        mtn=True
    )

    @property
    def agents(self):
        return gencfg_api.get_agents([self.gencfg_group])

    @classmethod
    def builder_config(cls, yt_observer, shard, full_build=False):
        if isinstance(shard, basestring):
            shard = tiers.parse_shard(shard)
        task = yt_observer.make_task(shard)
        if full_build:
            setattr(task, '_prev_shard_name', None)
        return {'shards': {shard.fullname: task.produce_config()}}

    def iter_host_port_config(self, yt_observer, shards, full_build=False):
        for agent, shard in zip(self.agents, shards):
            yield (agent.host, agent.port), self.builder_config(yt_observer, shard, full_build)


def _docs(obj, default=""):
    docs = (obj.__doc__ or default).strip()
    if '\n' in docs:
        docs = docs.partition('\n')[0]
    return docs.strip()


def _get_docs(user_ns):
    longest_var = max(len(key) for key in user_ns)
    variables = '\n'.join(
        '{} -> {}'.format(key.ljust(longest_var), _docs(user_ns[key]))
        for key in sorted(user_ns)
    )
    return "There are following predefined variables:\n{}\n".format(variables)


def _get_tracker(proxy='arnold', readonly=True):
    """tracker cli"""
    config = config_pb2.TServerConfig()
    config.Yt.Proxy = proxy
    config.Readonly = readonly
    config.ClientCount = 1
    return tracker.Tracker(config)


class YTCleaner(object):
    def __init__(self, proxy, yt_root, node_type, readonly=True):
        self._yt_client = yt_utils.create_yt_client(proxy)
        self._yt_root = yt_root
        self._node_type = node_type
        self._readonly = readonly

    def list_sub_nodes(self, path):
        """eg (/web/prod/) -> [/web/prod/1234/, /web/prod/1235/, ...]"""
        path = path.strip('/')
        paths = self._yt_client.search(
            os.path.join(self._yt_root, path),
            node_type=self._node_type,
            depth_bound=1,
        )
        result = {
            sub_node for sub_node in paths
            if sub_node != os.path.join(self._yt_root, path)
        }
        return sorted(result)

    def remove_node(self, node_path):
        assert node_path.startswith(self._yt_root)
        logging.warning('will remove %s', node_path)
        assert not self._readonly
        self._yt_client.remove(node_path, recursive=True)


def _get_tracker_cleanup_helper(proxy='arnold', readonly=True):
    """tracker cleanup helper"""
    return YTCleaner(proxy, tracker_table.YT_ROOT, 'map_node', readonly)


def _get_mappings_cleanup_helper(project, proxy='arnold', readonly=True):
    """mappings cleanup helper"""
    return YTCleaner(
        proxy,
        '//home/cajuper/user/{}/prod/chunks/'.format(project),
        None,
        readonly
    )


def main():
    logging.basicConfig(level=logging.DEBUG)
    user_ns = {
        'reports': ReportsKeepers(),
        'configs': ConfigsStorages(),
        'tiers': tiers,
        'entities': entities,
        'gencfg_api': gencfg_api,
        'yp_utils': yp_utils,
        'yt_utils': yt_utils,
        'bindings': bindings,
        'controllers': Controllers(),
        'user': user,
        'loop': loop,
        'root_ctrls': RootCtrls(registry.ROOT_CTRLS),
        'fast_builders': FastBuilders(),
        'get_tracker': _get_tracker,
        'get_tracker_cleanup_helper': _get_tracker_cleanup_helper,
        'get_mappings_cleanup_helper': _get_mappings_cleanup_helper,
    }

    cfg = traitlets.config.loader.Config()
    cfg.TerminalInteractiveShell.banner2 = _get_docs(user_ns)

    IPython.start_ipython(user_ns=user_ns, config=cfg)
