import logging
import contextlib

import porto
import retry
import six


class ContainerRunner(object):
    """
    Runs commands in porto container.
    Controls container by relative name.
    """
    def __init__(self, name, relative=True):
        self.short_name = name
        self.name = 'self/' + name if relative else name

    @retry.retry(delay=1, tries=3)
    def run(self, command, env=None, properties=None):
        self._destroy()
        self._run(command, env, properties or {})

    @property
    def respawn_count(self):
        with self._connect() as connection:
            return connection.GetProperty(self.name, 'respawn_count')

    @property
    def oom_kills(self):
        with self._connect() as connection:
            return connection.GetProperty(self.name, 'oom_kills')

    @property
    def is_running(self):
        with self._connect() as connection:
            try:
                return connection.GetProperty(self.name, 'state') == 'running'
            except porto.exceptions.ContainerDoesNotExist:
                _logger.warning('Container %s does not exist', self.name)
                return False

    def get_label(self, label):
        with self._connect() as connection:
            try:
                return connection.GetProperty(self.name, 'labels[{}]'.format(label))
            except porto.exceptions.Unknown:
                _logger.warning('Container %s has no label %s', self.name, label)
                return None

    def is_spawned(self):
        try:
            with self._connect() as connection:
                connection.Find(self.name)
                return True
        except porto.exceptions.ContainerDoesNotExist:
            return False

    def stop(self):
        self._destroy()

    def wait(self, timeout=None):
        try:
            with self._connect() as connection:
                connection.WaitContainers([self.name], timeout=timeout)
                _logger.info('joined container %s', self.name)
        except porto.exceptions.WaitContainerTimeout:
            _logger.info('container %s wait timeouted', self.name)

    def set_respawn(self, respawn=True):
        self.set_property('respawn', respawn)

    def set_property(self, name, value):
        with self._connect() as connection:
            connection.SetProperty(self.name, name, value)

    def _destroy(self):
        try:
            with self._connect() as connection:
                connection.Destroy(self.name)
                _logger.info('destroyed container %s', self.name)
        except porto.exceptions.ContainerDoesNotExist:
            pass

    def _run(self, command, env, properties):
        command = _format_command(command)
        with self._connect() as connection:
            container = connection.Create(self.name)

            for prop_name in properties:
                container.SetProperty(prop_name, properties[prop_name])

            container.SetProperty('command', command)
            container.SetProperty('respawn', True)
            container.SetProperty('env', _format_porto_env(env or {}))
            container.Start()
            _logger.info('spawned container %s', self.name)

    @contextlib.contextmanager
    def _connect(self):
        connection = None
        try:
            connection = porto.Connection()
            yield connection
        finally:
            if connection:
                connection.disconnect()

    def status(self):
        with self._connect() as connection:
            return connection.GetProperty(self.name, 'state')

    def get_command(self):
        with self._connect() as connection:
            return connection.GetProperty(self.name, 'command')


def _format_porto_env(env):
    porto_env = ''
    for key, value in env.items():
        porto_env += '{}={}; '.format(key, value)
    return porto_env


def _format_command(command):
    if isinstance(command, six.string_types):
        return command
    return ' '.join(str(field) for field in command)


_logger = logging.getLogger(__name__)
