import collections
import functools
import itertools
import logging
import time


class _ValueWrapper(object):
    __slots__ = ('value', 'use_till')

    def __init__(self, value, use_till=None):
        self.value = value
        self.use_till = use_till


class SaveWithTtl(object):
    __slots__ = ('value', 'ttl')

    def __init__(self, value, ttl):
        self.value = value
        self.ttl = ttl


class DontSave(object):
    __slots__ = ('value',)

    def __init__(self, value):
        self.value = value


def get_cache(records_limit):
    class Cache(object):
        _records_limit = records_limit

        def __init__(self, func):
            self._func = func
            self._cache = collections.OrderedDict()
            functools.update_wrapper(self, func)

        def _ensure_fresh(self, key):
            wrapped_value = self._cache.get(key, None)
            if wrapped_value and wrapped_value.use_till and (time.time() > wrapped_value.use_till):
                del self._cache[key]

        def __call__(self, *args, **kwargs):
            if self._records_limit is not None:
                if len(self._cache) > self._records_limit:
                    self._cleanup()

            key = _args_to_tuple(args, kwargs)
            self._ensure_fresh(key)

            if key not in self._cache:
                result = self._func(*args, **kwargs)
                if isinstance(result, DontSave):
                    return result.value
                elif isinstance(result, SaveWithTtl):
                    self._cache[key] = _ValueWrapper(result.value, time.time() + result.ttl)
                else:
                    self._cache[key] = _ValueWrapper(result)
            return self._cache[key].value

        def _cleanup(self):
            assert self._records_limit is not None
            # removes 500k records per second
            _log.debug('cleanup Cache (reached %s records)', len(self._cache))
            for key in list(itertools.islice(self._cache, 0, int(self._records_limit / 2))):
                del self._cache[key]
    return Cache


def _yield_from_each_it(*iterables):
    for it in iterables:
        for one in it:
            yield one


def _args_to_tuple(args, kwargs):
    if kwargs:
        return tuple(_yield_from_each_it(args, sorted(kwargs.items())))
    return args


memoized = get_cache(records_limit=None)

_log = logging.getLogger(__name__)
