from __future__ import absolute_import
import logging
import contextlib

import gevent
import retry
import porto
from six.moves import map


def execute(cmdline, working_dir, env=None, raise_on_error=True, name='bundle_build', properties=None):
    cmdline = ' '.join(cmdline)
    _log.debug('exec cmd %s', cmdline)
    _log.debug('exec env %s', env)
    _log.debug('exec properties %s', properties)
    with _create_subcontainer(name) as container:
        container.SetProperty('command', cmdline)
        container.SetProperty('stdout_path', name + '.stdout')
        container.SetProperty('stderr_path', name + '.stderr')
        container.SetProperty('cwd', working_dir)
        if env:
            container.SetProperty('env', _format_porto_env(env))
        if properties:
            list(map(container.SetProperty, list(properties.keys()), list(properties.values())))

        container.Start()
        _join(container)

        exit_status = container.GetProperty('exit_status')
        _log.debug('exit_status is %s', exit_status)
        if exit_status != 0 and exit_status != '0' and raise_on_error:
            raise RuntimeError('Process {} failed'.format(name))
        return exit_status


def _join(container):
    while True:
        try:
            if container.Wait(timeout=1):
                return
        except (porto.exceptions.SocketError, porto.exceptions.SocketTimeout):
            _log.warning('porto socket error')
        gevent.sleep(5)


def _format_porto_env(env):
    porto_env = ''
    for key, value in env.items():
        porto_env += '{}={}; '.format(key, value)
    return porto_env


@contextlib.contextmanager
def _create_subcontainer(name):
    conn = porto.Connection()
    conn.connect()

    container = _reclaim_container('self/' + name)
    try:
        yield container

    finally:
        container.Destroy()

    conn.disconnect()


@retry.retry(exceptions=porto.exceptions.EError, tries=10, delay=1)
def _reclaim_container(name):
    conn = porto.Connection()
    conn.connect()
    try:
        return conn.Create(name)
    except porto.exceptions.EError:
        conn.Destroy(name)
        raise


_log = logging.getLogger(__name__)
