import pytest

import infra.callisto.libraries.yt.tables as tables


def test_yt_composite_key_of_schema():
    single_key_schema = [
        {'name': 'the_only_key', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'not_key', 'type': 'string'},
    ]
    assert tables.composite_key_of_schema(single_key_schema) == ['the_only_key']

    composite_key_schema = [
        {'name': 'key_1', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'not_key', 'type': 'string'},
        {'name': 'key_2', 'type': 'string', 'sort_order': 'ascending'},
    ]
    assert tables.composite_key_of_schema(composite_key_schema) == ['key_1', 'key_2']

    no_key_schema = [
        {'name': 'not_key', 'type': 'string'},
        {'name': 'not_key_too', 'type': 'string'},
    ]

    with pytest.raises(ValueError):
        tables.composite_key_of_schema(no_key_schema)


def test_yt_composite_keys_schema():
    composite_key_schema = [
        {'name': 'key_1',   'type': 'string', 'sort_order': 'ascending'},
        {'name': 'key_2',   'type': 'string', 'sort_order': 'ascending'},
        {'name': 'not_key', 'type': 'string'},
    ]
    composite_key = tables.composite_key_of_schema(composite_key_schema)
    assert composite_key == ['key_1', 'key_2']
    a = [
        {'key_1': 1, 'key_2': 2, 'not_key': 3},
        {'key_1': 2, 'key_2': 2, 'not_key': 3},
        {'key_1': 2, 'key_2': 3, 'not_key': 4},
        {'key_1': 4, 'key_2': 4, 'not_key': 4},
        {'key_1': 4, 'key_2': 5, 'not_key': 4},
        {'key_1': 5, 'key_2': 4, 'not_key': 4},
    ]
    b = [
        {'key_1': 4, 'key_2': 4, 'not_key': 4},
    ]
    tables.ensure_unique_rows(a, composite_key)
    tables.ensure_unique_rows(b, composite_key)

    with pytest.raises(AssertionError):
        tables.ensure_unique_rows(a + b, composite_key)


def test_ensure_schema_is_ok():
    ok_schema = [
        {'name': 'key_1', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'key_2', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'not_key', 'type': 'string'},
    ]
    tables.ensure_schema_is_ok(ok_schema)

    not_ok_schema = [
        {'name': 'key_1', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'key_2', 'type': 'string', 'sort_order': 'ascending'},
        {'name': 'not_key', 'type': 'string'},
        {'name': 'key_2', 'type': 'string', 'sort_order': 'ascending'},
    ]
    with pytest.raises(ValueError):
        tables.ensure_schema_is_ok(not_ok_schema)
