import logging

import gevent
from . import vmtouch


def _eternal_loop(func, sleep_time):
    while True:
        try:
            func()
        except Exception as ex:
            logging.exception(ex.message)
        finally:
            gevent.sleep(sleep_time)


class Supervisor(object):
    def __init__(self, heater, warmness_threshold, target_path):
        self._heater = heater
        self._warmness_threshold = warmness_threshold
        self._target_path = target_path
        self._greenlet = None

    def start(self):
        if self._greenlet is None:
            self._greenlet = gevent.spawn(_eternal_loop, self._loop_iteration, 30)
        return self._greenlet

    def stop(self):
        if self._greenlet is not None:
            self._greenlet.kill()

    def get_warmness(self):
        warmness = vmtouch.in_memory_percent(self._target_path)
        logging.debug('warmness: %s, threshold: %s', warmness, self._warmness_threshold)
        return warmness

    def is_warmed(self):
        return self.get_warmness() >= self._warmness_threshold

    def _loop_iteration(self):
        if self.is_warmed():
            self._heater.ensure_not_heating()
        else:
            self._heater.ensure_heating()
