import logging
import contextlib

import infra.callisto.libraries.memoize as memoize
import yt.wrapper as yt
from .status_table import ObjectStatusTable
from .tables import SortedYtTable, OrderedYtTable, CachedStaticTable, gui_url, mount_table, create_yt_table

LockConflict = yt.errors.YtConcurrentTransactionLockConflict

__all__ = ['ObjectStatusTable', 'SortedYtTable', 'OrderedYtTable',
           'CachedStaticTable', 'gui_url', 'mount_table', 'create_yt_table',
           'create_yt_client', 'get_yt_client', 'ensure_node', 'lock',
           'wrap_strings', 'status_table', 'tables']


# yt_client.config cannot be resolved by IDEs
# noinspection PyUnresolvedReferences
def create_yt_client(proxy, use_rpc=False):
    yt_client = yt.YtClient(config=yt.config.config)
    yt_client.config['proxy']['url'] = proxy
    if use_rpc:
        yt_client.config['backend'] = 'rpc'
    return yt_client


@memoize.memoized
def get_yt_client(proxy, use_rpc=False):
    return create_yt_client(proxy, use_rpc=False)


def ensure_node(yt_client, path):
    if not yt_client.exists(path):
        yt_client.create('map_node', path)


@contextlib.contextmanager
def lock(yt_client, path, timeout_ms=None):
    with yt_client.Transaction(type='master', interrupt_on_failed=False) as transaction:
        lock_id = yt_client.lock(path, waitable=bool(timeout_ms), wait_for=timeout_ms)
        _log.debug('locked %s, lock_id: %s, transaction_id: %s', path, lock_id, transaction.transaction_id)
        yield transaction
    _log.debug('unlocked %s, lock_id: %s, transaction_id: %s', path, lock_id, transaction.transaction_id)


def wrap_strings(field_names):
    fields = ["'{}'".format(field) for field in field_names]
    return ', '.join(fields)


_log = logging.getLogger(__name__)
