import datetime
import logging

INSERT_MAX_ROW_COUNT = 100 * 1000


def create_yt_table(yt_client, path, **attributes):
    _log.info('will create table at %s', path)
    table = yt_client.TablePath(path)
    if attributes.get('tablet_cell_bundle', object()) is None:
        attributes.pop('tablet_cell_bundle', None)
    yt_client.create('table', table, attributes=attributes, recursive=True)


def mount_table(yt_client, path):
    _log.info('will mount table at %s', path)
    yt_client.mount_table(path)


def check_permissions(yt_client, path, read=False, write=False):
    if read:
        _log.debug('check %s read permissions of %s', yt_client.get_user_name(), path)
        assert yt_client.check_permission(yt_client.get_user_name(), 'read', path)['action'] == 'allow'
    if write:
        _log.debug('check %s write permissions of %s', yt_client.get_user_name(), path)
        assert yt_client.check_permission(yt_client.get_user_name(), 'write', path)['action'] == 'allow'


def composite_key_of_schema(schema):
    columns = []
    for column in schema:
        if 'sort_order' in column:
            columns.append(column['name'])
    if columns:
        return columns
    raise ValueError('no primary key found')


def ensure_unique_rows(rows, composite_key):
    all_keys = [
        composite_key_as_tuple(row, composite_key)
        for row in rows
    ]
    assert len(all_keys) == len(set(all_keys)), 'keys must be unique'


def composite_key_as_tuple(row, composite_key):
    return tuple(row[key] for key in composite_key)


def composite_key_as_row(row, composite_key):
    return {key: row[key] for key in composite_key}


def ensure_schema_is_ok(schema):
    column_names = [column['name'] for column in schema]
    if len(column_names) != len(set(column_names)):
        raise ValueError('multiple occurrence of columns with same name')


def gui_url(proxy, path):
    return 'https://yt.yandex-team.ru/{}/#page=navigation&path={}'.format(
        proxy.replace('.yt.yandex.net', ''), path.rstrip('/')
    )


class _YtTable(object):
    schema = None
    tablet_cell_bundle = 'cajuper'

    def __init__(self, yt_client, path, readonly=True):
        if self.schema:
            ensure_schema_is_ok(self.schema)
        self._yt_client = yt_client
        self._path = path
        self._readonly = readonly
        self._on_init_hook()

    def _on_init_hook(self):
        self.ensure_table()
        self.check_permissions()
        self.ensure_mounted()

    def ensure_table(self):
        if not _table_exists(self._yt_client, self._path):
            assert not self._readonly, '[{}] is in readonly mode'.format(self.path)
            create_yt_table(
                self._yt_client, self._path,
                dynamic=True,
                schema=self.schema,
                tablet_cell_bundle=self.tablet_cell_bundle,
            )

    def ensure_mounted(self):
        if not _table_mounted(self._yt_client, self._path):
            assert not self._readonly, '[{}] is in readonly mode'.format(self.path)
            mount_table(
                self._yt_client, self._path,
            )

    def check_permissions(self):
        check_permissions(self._yt_client, self._path, read=True, write=not self._readonly)

    @property
    def proxy(self):
        return self._yt_client.config['proxy']['url']

    @property
    def path(self):
        return self._path

    @property
    def gui_url(self):
        return gui_url(self.proxy, self.path)

    def __repr__(self):
        return '{}({})'.format(self.__class__.__name__, self.path)


class SortedYtTable(_YtTable):
    def __init__(self, yt_client, path, readonly=True):
        super(SortedYtTable, self).__init__(yt_client, path, readonly)
        self._composite_key = composite_key_of_schema(self.schema) if self.schema is not None else None

    def _select_rows(self, request, **kwargs):
        rows = []
        for row in self._yt_client.select_rows(request, **kwargs):
            rows.append(row)
        return rows

    def _lookup_rows(self, patterns):
        assert self._composite_key
        for pattern in patterns:
            assert not (set(pattern) - set(self._composite_key))
        rows = []
        for row in self._yt_client.lookup_rows(self.path, patterns):
            rows.append(row)
        return rows

    def _insert_rows(self, json_rows, format='json'):
        assert self._composite_key
        assert not self._readonly, 'readonly mode'
        ensure_unique_rows(json_rows, self._composite_key)
        self._yt_client.insert_rows(self._path, json_rows, format=format)

    def _delete_rows_by_request(self, request):
        assert self._composite_key
        assert not self._readonly, 'readonly mode'
        rows = self._select_rows(request)
        to_delete = [composite_key_as_row(row, self._composite_key) for row in rows]
        self._yt_client.delete_rows(self._path, to_delete, format='json')

    def _transaction(self):
        assert self._yt_client.config['backend'] == 'rpc', 'http fallback not currently supported'
        return self._yt_client.Transaction(type='tablet')

    def _insert_many_rows(self, json_rows, batch_size=INSERT_MAX_ROW_COUNT):
        """Useful when rows number is greater than 100k. Partial insert is not handled"""
        assert self._composite_key
        assert not self._readonly, 'readonly mode'
        ensure_unique_rows(json_rows, self._composite_key)

        batch_size = min(batch_size, INSERT_MAX_ROW_COUNT)
        for i in range(1 + len(json_rows) / batch_size):
            left, right = i * batch_size, (1 + i) * batch_size
            self._insert_rows(json_rows[left: right])


class OrderedYtTable(_YtTable):
    def _read_first_n(self, count):
        request = '* from [{}] where [$tablet_index] = 0 limit {}'.format(self.path, count)
        rows = []
        for row in self._yt_client.select_rows(request):
            rows.append(row)
        return rows

    def _trim_first_n(self, count):
        assert not self._readonly, 'readonly mode'
        request = '[$row_index] from [{}] where [$tablet_index] = 0 limit 1'.format(self.path)
        for row in self._yt_client.select_rows(request):
            self._yt_client.trim_rows(self.path, tablet_index=0, trimmed_row_count=row['$row_index'] + count)
            break

    def _insert_rows(self, json_rows):
        assert not self._readonly, 'readonly mode'
        self._yt_client.insert_rows(self._path, json_rows, format='json')


class CachedStaticTable(object):
    def __init__(self, path, yt_client, cache_ttl_minutes=5):
        self._path = path
        self._yt_client = yt_client
        self._yt_table = {}
        self._yt_access_time = datetime.datetime.min
        self._cache_ttl = datetime.timedelta(minutes=cache_ttl_minutes)

    def table(self):
        now = datetime.datetime.now()
        if now - self._yt_access_time > self._cache_ttl:
            self._yt_table = self._read_table()
            self._yt_access_time = now
        return self._yt_table

    def _read_table(self):
        return list(self._yt_client.read_table(self._path, format='json'))


def _table_mounted(yt_client, path):
    return yt_client.get_attribute(path, 'tablet_state') == 'mounted'


def _table_exists(yt_client, path):
    return yt_client.exists(path)


_log = logging.getLogger(__name__)
