import requests

from . import message as message_format
from . import report as report_format


class ReportNotParsed(RuntimeError):
    pass


WRITE_TIMEOUT = 10
READ_TIMEOUT = 30


class Client(object):
    _use_protobuf = False

    def __init__(self, url, min_uptime=0):
        self._url = url
        self._min_uptime = min_uptime

        if 'reports-v2-cajuper.n.yandex-team.ru' in url:
            self._use_protobuf = True

    def report(self, report):
        message = message_format.Message([report])
        self._report(message)

    def _report(self, message):
        url = self._url + '/report'

        if self._use_protobuf:
            content_type = 'application/protobuf'
            data = message_format.dump_protobuf(message)
        else:
            content_type = 'application/message'
            data = message_format.dump_msgpack(message)

        headers = {
            'Content-Type': content_type,
        }
        response = requests.post(
            url,
            data=data,
            timeout=WRITE_TIMEOUT,
            headers=headers,
        )
        if not response.ok:
            raise ReportNotParsed()

    def with_tags(self, tags):
        url = self._url + '/with_tags'
        headers = {
            'Accept': 'application/protobuf' if self._use_protobuf else 'application/message'
        }
        params = {'tag': list(tags), 'min_uptime': self._min_uptime}
        resp = requests.get(url, headers=headers, params=params, timeout=READ_TIMEOUT)
        if resp.status_code == 503:
            raise RuntimeError(resp.text)
        resp.raise_for_status()

        if self._use_protobuf:
            message = message_format.load_protobuf(resp.content)
        else:
            message = message_format.load_msgpack(resp.content)
        return message.reports

    @property
    def uptime(self):
        url = self._url + '/uptime'
        return int(requests.get(url, timeout=5).json()['uptime'])


class CachingClient(Client):
    def __init__(self, url, min_uptime=0):
        super(CachingClient, self).__init__(url, min_uptime=min_uptime)
        self._cache = {}

    def with_tags(self, tags):
        tags = frozenset(tags)
        if tags in self._cache:
            return self._cache[tags]
        superset_reports = _find_superset_reports_in_cache(self._cache, tags)
        if superset_reports is not None:
            self._cache[tags] = report_format.reports_with_tags(superset_reports, tags)
            return self._cache[tags]
        self._cache[tags] = super(CachingClient, self).with_tags(tags)
        return self._cache[tags]

    def drop_cache(self):
        self._cache = {}


def _find_superset_reports_in_cache(cache, tags):
    for key in cache:
        if tags.issuperset(key):
            return cache[key]
    return None
