import datetime

import msgpack

import report
import proto.report_pb2 as pb_module


class ParseMessageError(Exception):
    pass


class MessageVersionError(ParseMessageError):
    pass


class MessageFormatError(ParseMessageError):
    pass


class _Keys(object):
    Reports = 'reports'
    SendTime = 'send_time'
    Version = 'version'


class Message(object):
    version = 1

    def __init__(self, reports, send_time=None):
        self.reports = reports
        self.send_time = send_time or datetime.datetime.now()


def _datetime_to_timestamp(dt):
    return int(dt.strftime('%s'))


def _timestamp_to_datetime(timestamp):
    return datetime.datetime.fromtimestamp(timestamp)


def to_dict(message):
    return {
        _Keys.Version: message.version,
        _Keys.Reports: [report.to_dict(rep) for rep in message.reports],
        _Keys.SendTime: _datetime_to_timestamp(message.send_time),
    }


def from_dict(data):
    try:
        if data[_Keys.Version] == 1:
            return Message(
                [report.from_dict(rep) for rep in data[_Keys.Reports]],
                _timestamp_to_datetime(data[_Keys.SendTime]),
            )
        else:
            raise MessageVersionError
    except (KeyError, ValueError):
        raise MessageFormatError


def to_protobuf(message):
    assert isinstance(message, Message)
    pb_message = pb_module.TReportContainerMessage()
    pb_message.reports.extend([report.to_protobuf(rep) for rep in message.reports])
    pb_message.send_time = _datetime_to_timestamp(message.send_time)
    pb_message.version = message.version
    return pb_message


def from_protobuf(pb_message):
    assert pb_message.version == 1
    return Message(
        reports=[report.from_protobuf(rep) for rep in pb_message.reports],
        send_time=_timestamp_to_datetime(pb_message.send_time),
    )


def load_msgpack(data):
    return from_dict(msgpack.loads(data))


def dump_msgpack(message):
    """:type message: Message"""
    return msgpack.dumps(to_dict(message))


def load_protobuf(data):
    pb_message = pb_module.TReportContainerMessage()
    pb_message.ParseFromString(data)
    return from_protobuf(pb_message)


def dump_protobuf(message):
    pb_message = to_protobuf(message)
    return pb_message.SerializeToString()
