import json
import datetime

import msgpack

import proto.report_pb2 as pb_module


class ParseReportError(Exception):
    pass


class ReportVersionError(ParseReportError):
    pass


class ReportFormatError(ParseReportError):
    pass


class _Keys(object):
    Host = 'host'
    Port = 'port'
    Tags = 'tags'
    Data = 'data'
    Version = 'version'
    GenerationTime = 'time'
    Node = 'node'


class Report(object):
    version = 1

    __slots__ = ('host', 'port', 'tags', 'data', 'generation_time', 'node')

    def __init__(self, host, port, tags, data, generation_time=None, node=None):
        self.host = host
        self.port = int(port)
        self.tags = tags
        self.data = data
        self.generation_time = generation_time or datetime.datetime.now()
        self.node = node

    def match_tags(self, tags):
        """
        :type tags: set
        """
        return not (tags - self.tags)

    def __str__(self):
        return 'Report({}, {}, {}, {})'.format(self.host, self.port, list(self.tags), self.generation_time)


def reports_with_tags(reports, tags):
    return [
        report for report in reports
        if not tags or report.match_tags(tags)
    ]


def _datetime_to_timestamp(dt):
    return int(dt.strftime('%s'))


def _timestamp_to_datetime(timestamp):
    return datetime.datetime.fromtimestamp(timestamp)


def to_dict(report):
    return {
        _Keys.Host: report.host,
        _Keys.Port: report.port,
        _Keys.Tags: list(report.tags),
        _Keys.Data: report.data,
        _Keys.GenerationTime: _datetime_to_timestamp(report.generation_time),
        _Keys.Version: report.version,
        _Keys.Node: report.node,
    }


def from_dict(dct):
    try:
        if dct[_Keys.Version] == 1:
            return Report(
                dct[_Keys.Host],
                dct[_Keys.Port],
                set(dct[_Keys.Tags]),
                dct[_Keys.Data],
                _timestamp_to_datetime(dct[_Keys.GenerationTime]),
                dct.get(_Keys.Node)
            )
        else:
            raise ReportVersionError
    except (KeyError, ValueError):
        raise ReportFormatError


def to_protobuf(report):
    assert isinstance(report, Report)
    pb_report = pb_module.TReportMessage()
    pb_report.host = report.host
    pb_report.port = report.port
    pb_report.tags.extend(report.tags)
    pb_report.data = msgpack.dumps(report.data)
    pb_report.timestamp = _datetime_to_timestamp(report.generation_time)
    pb_report.version = report.version
    pb_report.node = report.node
    return pb_report


def from_protobuf(pb_report):
    assert pb_report.version == 1
    return Report(
        host=pb_report.host,
        port=pb_report.port,
        tags=set(pb_report.tags),
        data=msgpack.loads(pb_report.data),
        generation_time=_timestamp_to_datetime(pb_report.timestamp),
        node=pb_report.node
    )


def dump_human_readable(report):
    meta = '\n'.join((
        'Host: {}'.format(report.host),
        'Port: {}'.format(report.port),
        'Node: {}'.format(report.node),
        'Tags: [{}]'.format(', '.join(report.tags)),
        'Time: {}'.format(report.generation_time),
        'Version: {}'.format(report.version),
    ))

    try:
        content = json.dumps(report.data, indent=4)
    except TypeError:
        content = str(report.data or '')
    return '{}\n\n{}'.format(meta, content)


def dump_json(report):
    return json.dumps(to_dict(report))
