import json
import logging
import collections

import werkzeug.routing as routing
import werkzeug.wrappers as wrappers
import infra.callisto.reports.message as message_module
import infra.callisto.reports.report as report_module

import consumer


class Application(object):
    def __init__(self, consumer_):
        self._consumer = consumer_

        self._url_map = routing.Map([
            routing.Rule('/report/<host>/<int:port>', endpoint=self._get_report, methods=['GET']),
            routing.Rule('/reports', endpoint=self._get_reports, methods=['GET']),
            routing.Rule('/with_tags', endpoint=self._get_reports, methods=['GET']),
            routing.Rule('/uptime', endpoint=self._get_uptime, methods=['GET']),

            routing.Rule('/info/requests_count', endpoint=self._get_requests_count, methods=['GET']),
        ])

        self._plain_map = {
            '/report': self._post_report,
        }

        self._requests_counter = collections.defaultdict(int)

    def __call__(self, environ, start_response):
        url = environ['PATH_INFO']
        if url in self._plain_map:
            result = self._plain_map[url](environ, start_response)
        else:
            request = wrappers.Request(environ)
            urls = self._url_map.bind_to_environ(environ)

            result = urls.dispatch(
                lambda endpoint, kwargs: endpoint(request, **kwargs),
                catch_http_exceptions=True
            )(environ, start_response)

        self._requests_counter[url] += 1
        return result

    def _post_report(self, environ, start_response):
        try:
            data = environ['wsgi.input'].read()
            if environ.get('CONTENT_TYPE') == 'application/protobuf':
                message = message_module.load_protobuf(data)
            else:
                message = message_module.load_msgpack(data)
            self._consumer.consume_reports(message.reports)
            start_response('200 OK', (('Content-Type', 'text'),))
            return ()
        except message_module.ParseMessageError:
            logging.warn('not parsed message')
        except report_module.ParseReportError:
            logging.warn('not parsed report')

        start_response('400 Bad Request', (('Content-Type', 'text'),))
        return ('not parsed',)

    def _get_report(self, request, host, port):
        report = self._consumer.get(host, port)
        if not report:
            return wrappers.Response('report not found', 404)
        return wrappers.Response(report_module.dump_human_readable(report), content_type='text')

    def _get_reports(self, request):
        mime_type = request.accept_mimetypes.best_match((
            'application/json', 'application/message', 'application/protobuf',
        ))
        tags = set(request.args.getlist('tag'))

        min_uptime = float(request.args.get('min_uptime', 0))
        if self._consumer.uptime < min_uptime:
            return wrappers.Response(
                'not enough uptime: {}, requested {}'.format(
                    self._consumer.uptime,
                    min_uptime,
                ), status=503
            )

        reports_with_tags = self._consumer.with_tags(tags)
        message = message_module.Message(reports_with_tags)

        if 'message' in mime_type:
            data = message_module.dump_msgpack(message)
        elif 'protobuf' in mime_type:
            data = message_module.dump_protobuf(message)
        else:
            mime_type = 'application/json'
            data = json.dumps({
                '{}:{}'.format(report.host, report.port): report_module.to_dict(report)
                for report in reports_with_tags
            }, indent=4)
        return wrappers.Response(data, 200, content_type=mime_type)

    def _get_uptime(self, request):
        return wrappers.Response(json.dumps({
            'uptime': self._consumer.uptime
        }), status=200)

    def _get_requests_count(self, request):
        return wrappers.Response(json.dumps(self._requests_counter), status=200)


def create_app():
    return Application(consumer.ReportsConsumer())
