#include "handlers.h"


namespace NReportsKeeper::NHandlers {
    using TReportContainer = infra::callisto::TReportContainerMessage;

    ui64 GetMinUptime(const NBubbleServer::TRequest &request) {
        return IntFromString<ui64, 10>(request.GetCgiFirst("min_uptime", "0"));
    }

    TString DumpReportsAsProtobuf(const TVector<TReportRef>& reports) {
        TReportContainer container;
        container.set_version(1);
        container.mutable_reports()->Reserve(static_cast<int>(reports.size()));
        for (const auto& report : reports) {
            *(container.add_reports()) = *report;
        }
        return container.SerializeAsString();
    }

    TString DumpReportsHumanReadable(const TVector<TReportRef>& reports) {
        TStringStream stream;
        for (const auto& report : reports) {
            stream << report->DumpHumanReadable() << "\n\n";
        }
        return stream.Str();
    }

    THttpResponse TWithTags::Execute(const NBubbleServer::TRequest &request) {
        TSet<TString> tags = request.GetCgi("tag");
        auto min_uptime = TDuration::Seconds(GetMinUptime(request));
        if (Storage().SinceFirstReport().GetValue() < min_uptime.GetValue()) {
            TStringStream stream;
            stream << "not enough uptime: "
                   << Storage().SinceFirstReport().Seconds()
                   << ". Requested " << min_uptime;
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_SERVICE_UNAVAILABLE, stream.Str());
        }
        if (request.Accepts("application/protobuf")) {
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_OK, DumpReportsAsProtobuf(Storage().WithTags(tags)));
        } else {
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_OK, DumpReportsHumanReadable(Storage().WithTags(tags)));
        }
    }


    TInstance GetHostPort(const TStringBuf &path) {
        try {
            TVector<TStringBuf> parts;
            Split(path, "/", parts);
            if (parts.size() != 3) {
                ythrow yexception();
            }
            return {TString{parts[1]}, IntFromString<ui64, 10>(parts[2])};
        } catch (const yexception& exc) {
            ythrow yexception() << "Cannot parse path: " << path << "\n"
                                << "Expected: /reports/<host>/<int:port>";
        }
    }

    THttpResponse TGetReport::Execute(const NBubbleServer::TRequest &request) {
        auto report = Storage().GetReport(GetHostPort(request.Path));
        if (report != nullptr) {
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_OK, report->DumpHumanReadable());
        } else {
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_NOT_FOUND, "Report not found");
        }
    }


    THttpResponse TConsumeReport::Execute(const NBubbleServer::TRequest &request) {
        TReportContainer container;
        if (!container.ParseFromArcadiaStream(&request.Input)) {
            Cerr << "Could not parse report\n";
            return NBubbleServer::MakeResponse(HttpCodes::HTTP_BAD_REQUEST, "Could not parse report");
        }
        for (const auto& report : container.Getreports()) {
            TReportRef report_ref(new TReport(report));
            Storage().ConsumeReport(report_ref);
        }
        return NBubbleServer::MakeResponse(HttpCodes::HTTP_OK);
    }
}

