#include <util/stream/str.h>
#include <util/string/join.h>
#include <contrib/libs/msgpack/include/msgpack.hpp>
#include <util/datetime/systime.h>
#include <util/datetime/base.h>
#include <sstream>
#include "storage.h"

namespace NReportsKeeper {
    TString msgpack_to_json(const TString& data) {
        auto oh = msgpack::unpack(data.data(), data.size());
        std::stringstream sstream;
        sstream << oh.get();
        return TString(sstream.str());
    }

    const TInstance& TReport::Instance() const {
        return Instance_;
    }

    const TSet<TString>& TReport::Tags() const {
        return Tags_;
    }

    bool TReport::MatchTags(const TSet<TString>& tags) const {
        const TSet<TString>& self_tags = Tags();
        for (const TString& tag : tags) {
            if (!self_tags.contains(tag))
                return false;
        }
        return true;
    }

    TString TReport::DumpHumanReadable() const {
        TStringStream stream;
        stream << "Host: " << Gethost() << "\n"
               << "Port: " << Getport() << "\n"
               << "Node: " << Getnode() << "\n"
               << "Time: " << Gettimestamp() << "\n"
               << "Tags: [" << JoinSeq(", ", Tags()) << "]" << "\n"
               << "Version: " << Getversion() << "\n"
               << "Content: " << "\n";
        stream << msgpack_to_json(Getdata());
        return stream.Str();
    }

    void TStorage::ConsumeReport(TReportRef report) {
        const auto& instance = report->Instance();
        TWriteGuard lock(Mutex_);
        Dict_[instance] = report;
        if (!FirstReportConsumed_.GetValue()) {
            FirstReportConsumed_ = Now();
        }
    }

    TDuration TStorage::SinceFirstReport() const {
        if (!FirstReportConsumed_.GetValue())
            return {};
        return Now() - FirstReportConsumed_;
    }

    TReportRef TStorage::GetReport(const TInstance& instance) {
        TReadGuard lock(Mutex_);
        if (Dict_.contains(instance))
            return Dict_.at(instance);
        return TReportRef(nullptr);
    }

    TVector<TReportRef> TStorage::WithTags(const TSet<TString>& tags) {
        TVector<TReportRef> result;
        TReadGuard lock(Mutex_);
        for (const auto& kv : Dict_) {
            if (kv.second->MatchTags(tags)) {
                result.emplace_back(kv.second);
            }
        }
        return result;
    }
}
