import os
import stat
import time
import shutil
import tarfile
import logging
import datetime

import retry

import infra.callisto.deploy.tracker.client as tracker
import infra.callisto.libraries.process as process

import external
import utils
import enums
import bundle_build_function


BUNDLE_LOGS_TTL = 7 * 24 * 3600
FAILURE_AFTER_START_TOLERANCE = 60 * 30


def build(shard, shards_storage, aux_storage, logs_dir, prev_shard_path=None):
    started_at = time.time()
    try:
        args = shard.config['args']

        build_dir = shards_storage.get_slot(shard.name, build=True)

        bundle_dir = aux_storage.get_resource(shard.bundle, max_dl_speed='10M')
        _untar_all(bundle_dir)
        entry_point = os.path.join(bundle_dir, args['entry_point'])
        _ensure_executable(entry_point)

        working_dir = os.path.join(logs_dir, shard.name, _logs_dir_name())
        utils.ensure_dir(working_dir)
        _cleanup_working_dirs(logs_dir)

        config = aux_storage.get_resource(args['bundle_config']['rbtorrent']) if 'bundle_config' in args else None

        env = utils.prepare_env(args.get('env'))

        kwargs = dict(
            args,
            entry_point=entry_point,
            bundle_dir=bundle_dir,
            output_dir=build_dir,
            config=config,
            working_dir=working_dir,
            shard_name=shard.name,
            prev_shard_path=prev_shard_path,
            env=env,
        )
        bundle_build_function.bundle_build(args['bundle']['type'], **kwargs)

        if args.get('configure', False):
            _configure_shard(build_dir, shard.name)
        elif args.get('configure_fast', False):
            _configure_shard_fast(build_dir, shard.name, shards_storage.share(shard.name))
        elif args.get('configure_inc', False):
            _configure_inc(build_dir, shard.name, args.get('required_shard_id'))

        rbtorrent_in_share = shards_storage.share(shard.name)
        logging.info('%s rbtorrent = %s', shard.name, rbtorrent_in_share)

        if args.get('register') and isinstance(args.get('register'), dict):
            if args['register'].get('attribute_shard'):
                _share_and_register_webtier0(build_dir, shard.name, args['register'])
            else:
                if args['register']['cajuper']:
                    _share_and_register_cajuper(build_dir, args['register']['tracker_url'],
                                                args['register']['namespace'],
                                                args['register'].get('recursive_share_depth', 5), exclude=None)
                if args['register']['iss']:
                    _ensure_registered(build_dir, shard.name, rbtorrent_in_share)
        elif args.get('register'):
            _ensure_registered(build_dir, shard.name, rbtorrent_in_share)

        shards_storage.set_ready(shard.name, build=True, rbtorrent=rbtorrent_in_share)
        shard.status = enums.Status.DONE
        shard.set_checked()
    except Exception:
        logging.exception('Exception')
        shard.on_failed_build()
        if time.time() - started_at < FAILURE_AFTER_START_TOLERANCE or shard.attempt == 1:
            shard.status = enums.Status.IDLE
        else:
            shard.status = enums.Status.FAILURE
            logging.info('failed after %s seconds', time.time() - started_at)
        raise


def _untar_all(bundle_dir):
    dst_path = os.path.join(bundle_dir, 'JUPITER_BUNDLE')

    for f in os.listdir(bundle_dir):
        file_path = os.path.join(bundle_dir, f)
        if os.path.isfile(file_path) and tarfile.is_tarfile(file_path):
            with tarfile.open(file_path) as tar:
                tar.extractall(path=dst_path)


def _ensure_executable(path):
    os.chmod(path, os.stat(path).st_mode | stat.S_IEXEC)


def _configure_shard(shard_dir, shard_id):
    process.execute(['./iss_shards', 'configure', '--id', shard_id, shard_dir], raise_on_error=True)


def _configure_inc(shard_dir, shard_id, required_shard_id=None):
    args = [
        './iss_shards', 'configure',
        '--id', shard_id,
    ]
    if required_shard_id:
        args += [
            '--required-id', required_shard_id,
        ]
    args += [
        '--install-script', './install.sh',
        shard_dir
    ]
    process.execute(args, raise_on_error=True)


def _configure_shard_fast(shard_dir, shard_id, rbtorrent):
    shard_conf = [
        '%shard {}'.format(shard_id),
        '%mtime 1400000000',
        '%files',
    ] + [
        '%attr(md5={md5}, nocheckmtime, size={size}) {name}'.format(
            md5=f['md5sum'],
            size=f['size'],
            name=f['name'],
        )
        for f in utils.sky_files(rbtorrent)
    ]
    shard_conf = '\n'.join(shard_conf + [''])
    with open(os.path.join(shard_dir, 'shard.conf'), 'w') as f:
        f.write(shard_conf)


@retry.retry(tries=10, delay=10, max_delay=300, backoff=2)
def _register_shard(shard_dir):
    process.execute(['./iss_shards', 'register', shard_dir], raise_on_error=True)


def _ensure_registered(shard_dir, shard_name, rbtorrent_in_share):
    rbtorrent_in_tracker = external.find_rbtorrent(shard_name)
    logging.debug('local rb: %s, in tracker: %s', rbtorrent_in_share, rbtorrent_in_tracker)
    if rbtorrent_in_tracker is None:
        _register_shard(shard_dir)
    elif rbtorrent_in_tracker != rbtorrent_in_share:
        if external.is_registered_inc(shard_name):
            logging.info('shard was registered as incremental, try to register again to ensure rbtorrent is ok')
            _register_shard(shard_dir)
        else:
            raise RuntimeError('shard rbtorrent differs from one in tracker')


def _cleanup_working_dirs(working_root_dir):
    for dir_entity in os.listdir(working_root_dir):
        fullpath_entity = os.path.join(working_root_dir, dir_entity)

        if not os.path.isdir(fullpath_entity):
            continue

        if os.stat(fullpath_entity).st_mtime < time.time() - BUNDLE_LOGS_TTL:
            shutil.rmtree(fullpath_entity)


def _cleanup_target_dir(target_dir):
    if os.path.exists(target_dir):
        utils.remove_dir(target_dir)
    utils.recreate_dir(target_dir)


def _logs_dir_name():
    return datetime.datetime.now().isoformat('_').replace(':', '-')


def _share_and_register_cajuper(build_dir, tracker_url, namespace, max_depth, exclude=None, include=None, replace_name=None):
    resources = tracker.share_recursive(build_dir, max_depth, exclude, include)

    # if there is RbTorrentDiffers exception, the loop will break immediately
    @retry.retry(IOError, tries=10000, delay=5, backoff=1.5, max_delay=60)
    def _do_request():
        client = tracker.Client(tracker_url)
        request = client.make_register_request(namespace)
        for resource in resources:
            request.add_resource(
                name=replace_name(resource.path) if replace_name else resource.path,
                rbtorrent=resource.rbtorrent,
                size=resource.size,
            )
        request.register()

    _do_request()


def _share_and_register_webtier0(build_dir, shard_name, register_args):
    # Register primary shard with excluded files.
    _share_and_register_cajuper(build_dir,
                                register_args['tracker_url'], register_args['namespace'],
                                register_args.get('recursive_share_depth', 5),
                                exclude='*/indexkeyinv_global_ranges/*')

    # Register attribute shard with excluded files and with name replacements and in different namespace.
    attribute_args = register_args['attribute_shard']
    _share_and_register_cajuper(build_dir,
                                register_args['tracker_url'], attribute_args['namespace'],
                                register_args.get('recursive_share_depth', 5),
                                replace_name=lambda name: name.replace(shard_name, attribute_args['shard_name']),
                                exclude='*/indexkeyinv_global_ranges/*')

    # Register mappings shard with included files and with name replacements and in different namespace.
    mapping_args = register_args.get('mapping_shard')
    if mapping_args:
        _share_and_register_cajuper(build_dir,
                                    register_args['tracker_url'], mapping_args['namespace'],
                                    register_args.get('recursive_share_depth', 5),
                                    replace_name=lambda name: name.replace(shard_name, mapping_args['shard_name']),
                                    include='*mapping*')
