import logging
import os

import porto_executor
import utils


def _ensure_no_hidden_dirs(directory):
    for sub_dir in os.listdir(directory):
        if os.path.isdir(os.path.join(directory, sub_dir)):
            assert not sub_dir.startswith('.'), 'found hidden dir {}'.format(sub_dir)


def _ensure_not_empty(directory):
    assert len(os.listdir(directory)) > 0


def _build_jupiter_shard(
    entry_point,
    working_dir,
    output_dir,
    env,
    yt_state, number, tier, yt_path, config, prev_shard_path, **kwargs
):
    args = [
        entry_point,
        '-o', output_dir,
        '-g', yt_state,
        '-s', number,
        '-t', tier,
        '-p', yt_path,
        '-m', working_dir,
    ]
    if config:
        args += ['-c', config]
    if prev_shard_path:
        args += ['-r', prev_shard_path]
    if 'yt_proxy' in kwargs:
        args += ['-h', kwargs['yt_proxy']]
    if 'make_file' in kwargs:
        args += ['-f', kwargs['make_file']]

    if kwargs.get('extra_args'):
        args += kwargs['extra_args']
    cmdline = ' '.join(args)
    porto_executor.execute(cmdline, working_dir, env)
    _ensure_no_hidden_dirs(output_dir)
    _ensure_not_empty(output_dir)


def _build_adv_machine_shard(
    entry_point,
    working_dir,
    output_dir,
    env,
    bundle_dir, number, tier, yt_path, config_point, data_point, **kwargs
):
    args = [
        entry_point,
        'build-shard',
        '-s', 'hahn',
        '--config-dir', os.path.join(bundle_dir, config_point),
        '--data-dir', os.path.join(bundle_dir, data_point),
        '--output-index-dir', output_dir,
        '--tmp-output-index-dir', output_dir,
        '--input-prefix', yt_path,
        '--name', tier,
        '--shard-id', number,
        '-S', '1',
        '-vv',
    ]
    cmdline = ' '.join(args)
    porto_executor.execute(cmdline, working_dir, env)


def _build_img_thumb_delta(
    entry_point,
    working_dir,
    output_dir,
    shard_name,
    yt_proxy,
    env,
):
    logging.debug('start _build_img_thumb_delta')
    utils.ensure_dir(working_dir)
    env = dict(
        env,
        YT_PREFIX='//home/',
        BSCONFIG_SHARDNAME=shard_name,
        SHARD_PREFIX=output_dir,
        MR_SERVER=yt_proxy,
        # BASE_PREFIX='images/taas',  temporarily arrives from controller. FIXME: use yt_path instead
    )
    porto_executor.execute(entry_point, working_dir, env)
    logging.debug('done _build_img_thumb_delta')


def _merge_img_thumb_shard(
    entry_point,
    working_dir,
    output_dir,
    path_to_lthdb,
    prev_shard_path, number,
    env,
):
    logging.debug('start _merge_img_thumb_shard')
    utils.ensure_dir(working_dir)
    args = [
        entry_point, 'merge-supervised',
        '--shard-number', number,
        '--database-name', 'thdb',
        '--database-path', output_dir,
        '--new-database-path', path_to_lthdb,
        '--portion-map', os.path.join(path_to_lthdb, 'portion.map'),
        '--alive', os.path.join(path_to_lthdb, 'thdb.alive'),
    ]
    if prev_shard_path:
        args += ['--prev-database-path', prev_shard_path]
    cmd = ' '.join(args)
    porto_executor.execute(cmd, working_dir, env)
    logging.debug('done _merge_img_thumb_shard')


# def _build_img_thumb_shard(
#     working_dir,
#     bundle_dir,
#     output_dir,
#     env,
#     prev_shard_path, shard_name, number, **kwargs
# ):
#     bundle_dir = os.path.join(bundle_dir, 'imgsth')
#     path_to_delta = os.path.join(output_dir, 'delta')
#
#     utils.ensure_dir(path_to_delta)
#     _build_img_thumb_delta(
#         entry_point=os.path.join(bundle_dir, 'shard_download'),
#         working_dir=os.path.join(working_dir, 'delta'),
#         output_dir=path_to_delta,
#         shard_name=shard_name,
#         yt_proxy=kwargs.get('yt_proxy', 'banach.yt.yandex.net'),
#         env=env,
#     )
#
#     _merge_img_thumb_shard(
#         entry_point=os.path.join(bundle_dir, 'thdb'),
#         working_dir=os.path.join(working_dir, 'merge'),
#         output_dir=output_dir,
#         path_to_lthdb=os.path.join(path_to_delta, shard_name),
#         prev_shard_path=prev_shard_path,
#         number=number,
#         env=env,
#     )
#     utils.remove_dir(path_to_delta)
#     logging.debug('removed delta path %s', path_to_delta)


def _build_img_thumb_shard(
    working_dir,
    bundle_dir,
    output_dir,
    env,
    shard_name, **kwargs
):
    bundle_dir = os.path.join(bundle_dir, 'imgsth')

    _build_img_thumb_delta(
        entry_point=os.path.join(bundle_dir, 'shard_download'),
        working_dir=working_dir,
        output_dir=os.path.dirname(output_dir),
        shard_name=shard_name,
        yt_proxy=kwargs.get('yt_proxy', 'banach.yt.yandex.net'),
        env=env,
    )
    utils.ensure_dir(os.path.join(output_dir, 'bin'))
    utils.copy_file(os.path.join(bundle_dir, 'thdb'), os.path.join(output_dir, 'bin', 'thdb'))
    utils.copy_file(os.path.join(bundle_dir, 'install.sh'), os.path.join(output_dir, 'install.sh'))
    os.chmod(os.path.join(output_dir, 'install.sh'), 0775)
    os.chmod(os.path.join(output_dir, 'bin', 'thdb'), 0775)


# Left untouched for imgs tier1 experiment.
def _build_img_shard(
    working_dir, bundle_dir, output_dir,
    env, shard_name, entry_point, **kwargs
):
    utils.ensure_dir(working_dir)
    utils.create_symlink(bundle_dir, os.path.join(working_dir, 'bundle'))
    env = dict(
        env,
        YT_PREFIX='//home/',
        BSCONFIG_SHARDNAME=shard_name,
        SHARD_PREFIX=os.path.dirname(output_dir),
        MR_SERVER=kwargs.get('yt_proxy', 'banach.yt.yandex.net'),
    )
    porto_executor.execute(os.path.join(bundle_dir, entry_point), working_dir, env)


def _build_img_rim_shard(
    working_dir, bundle_dir, output_dir,
    env, shard_name, **kwargs
):
    utils.ensure_dir(working_dir)
    utils.create_symlink(bundle_dir, os.path.join(working_dir, 'bundle'))

    env['BSCONFIG_SHARDNAME'] = shard_name
    env['SHARD_PREFIX'] = os.path.dirname(output_dir)
    env['MR_RUNTIME'] = 'YT'
    env.setdefault('MR_SERVER', kwargs.get('yt_proxy', 'banach.yt.yandex.net'))

    porto_executor.execute(os.path.join(bundle_dir, 'imgsrim', 'shard_download'), working_dir, env)


def _build_img_commercial_shard(
    working_dir, bundle_dir, output_dir,
    env, shard_name, **kwargs
):
    utils.ensure_dir(working_dir)
    utils.create_symlink(bundle_dir, os.path.join(working_dir, 'bundle'))

    env['BSCONFIG_SHARDNAME'] = shard_name
    env['SHARD_PREFIX'] = os.path.dirname(output_dir)
    env['MR_RUNTIME'] = 'YT'
    env.setdefault('MR_SERVER', kwargs.get('yt_proxy', 'banach.yt.yandex.net'))

    env['BASE_PREFIX'] = '//home/images-money'
    env['TIER_NAME'] = 'ImgCommercialTier0'

    porto_executor.execute(os.path.join(bundle_dir, 'imgs_commercial_data', 'shard_download'), working_dir, env)


_bundle_build_functions = {
    'adv_machine': _build_adv_machine_shard,
    'callisto': _build_jupiter_shard,
    'jupiter': _build_jupiter_shard,
    'video': _build_jupiter_shard,
    'img_thumb': _build_img_thumb_shard,
    'img': _build_img_shard,
    'img_rim': _build_img_rim_shard,
    'img_commercial': _build_img_commercial_shard,
}


def bundle_build(bundle_type, **kwargs):
    func = _bundle_build_functions[bundle_type]
    func(**kwargs)
