import argparse
import gevent
import gevent.monkey
import json

import infra.callisto.libraries.logging as cajuper_logging

from server import serve_forever
from shardtool2 import ShardTool2


def run(options):
    try:
        max_inc_tries = 2 ** 64 if options.max_inc_tries == 'infinity' else int(options.max_inc_tries)
        shardtool = ShardTool2(
            '{}/{}'.format(options.shard_root, options.port),
            options.shard_root,
            '{}/{}.aux'.format(options.shard_root, options.port),
            options.max_dl_speed,
            options.max_ul_speed,
            options.copier_opts,
            options.logs_dir,
            max_inc_tries,
            options.recheck_shards,
        )

        if options.config:
            shardtool.reconfigure(json.load(open(options.config)).get('shards'))

        gevent.spawn(serve_forever, shardtool.get_status, shardtool.reconfigure, ('::', options.port), shardtool)

        shardtool.run()
    except KeyboardInterrupt:
        pass


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--port', help='Server port', type=int, required=True)
    parser.add_argument('--shard-root', help='Root directory for shards', default='.', required=False)
    parser.add_argument('--debug', help='Debug mode (affects logging)', action='store_true', default=False)
    parser.add_argument('--max-dl-speed', help='Max download speed for skynet', type=str, required=False)
    parser.add_argument('--max-ul-speed', help='Max upload speed for skynet', type=str, required=False)
    parser.add_argument('--copier-opts', help='Extra options for sky get (sky get --opts)', type=str, required=False)
    parser.add_argument('--config', help='Initial config', type=str, required=False)
    parser.add_argument('--logs-dir', help='Logs directory', type=str, required=False, default='./logs')
    parser.add_argument('--max-inc-tries', help='max inc attempts before full build (int or infinity)', default=1)
    parser.add_argument('--recheck-shards', help='recheck shards to avoid broken', action='store_true', default=False)

    return parser.parse_args()


def main():
    gevent.monkey.patch_all(dns=False)

    args = parse_args()
    cajuper_logging.configure_logging(path_to_logs=args.logs_dir, debug=args.debug)
    run(args)
