import os
import logging
import contextlib

import gevent
import retry
import porto
import library.python.resource as resource

import external


def execute(cmdline, working_dir, env=None, raise_on_error=True):
    logging.debug('exec cmd %s', cmdline)
    logging.debug('exec env %s', env)
    with _create_subcontainer('bundle_build') as container:
        container.SetProperty('command', _redirect_std_streams(cmdline, working_dir))
        container.SetProperty('stdout_path', '/dev/null')
        container.SetProperty('stderr_path', '/dev/null')
        container.SetProperty('cwd', working_dir)
        if env:
            container.SetProperty('env', _format_porto_env(env))

        container.Start()
        _join(container)

        exit_status = container.GetProperty('exit_status')
        logging.debug('exit_status is %s', exit_status)
        if exit_status != 0 and exit_status != '0' and raise_on_error:
            raise external.SubprocessException((_redirect_std_streams(cmdline, working_dir),))
        return exit_status


def _join(container):
    while True:
        try:
            if container.Wait(timeout=1):
                return
        except (porto.exceptions.SocketError, porto.exceptions.SocketTimeout):
            logging.warning('porto socket error')
        gevent.sleep(5)


def _format_porto_env(env):
    porto_env = ''
    for key, value in env.items():
        porto_env += '{}={}; '.format(key, value)
    return porto_env


def _redirect_std_streams(cmdline, working_dir):
    abs_working_dir = os.path.abspath(working_dir)
    with open(os.path.join(abs_working_dir, 'redirect_std_streams.sh'), 'w') as fp:
        fp.write(resource.find('/redirect_std_streams.sh'))
        fp.close()

    return '/bin/sh {working_dir}/redirect_std_streams.sh {cmdline}'.format(
        cmdline=cmdline,
        working_dir=abs_working_dir
    )


@contextlib.contextmanager
def _create_subcontainer(name):
    conn = porto.Connection()
    conn.connect()

    container = _reclaim_container('self/' + name)
    try:
        yield container

    finally:
        container.Destroy()

    conn.disconnect()


@retry.retry(exceptions=porto.exceptions.EError, tries=10, delay=1)
def _reclaim_container(name):
    conn = porto.Connection()
    conn.connect()
    try:
        return conn.Create(name)
    except porto.exceptions.EError:
        conn.Destroy(name)
        raise


def _test(shard):
    print '_test', shard
    # exec2('stress --cpu 1', raise_on_error=True)
    execute('/bin/bash -c "sleep 1; exit 1"', '.', raise_on_error=True)


def _main():
    import gevent.monkey
    gevent.monkey.patch_all()

    from shardtool2 import Worker
    worker = Worker(fn=_test, shard='hi there')
    worker.greenlet.start()
    print 'running'
    gevent.sleep(5)
    print 'stopping'
    worker.stop()
    print 'stopped'
