import logging
import json
import urlparse

import os
from gevent.pywsgi import WSGIServer


def serve_forever(status_callback, reload_callback, endpoint, shardtool=None):
    def serve_request(env, start_response):
        targets_dir = shardtool.logs_dir
        path = env['PATH_INFO']

        if path in ['/status', '/status_message']:  # /status_message for old instancestate compatibility
            start_response('200 OK', [('Content-Type', 'application/json')])
            return [json.dumps(status_callback(), indent=4)]

        elif path in ['/reload']:
            reload_callback(json.load(env['wsgi.input'])['shards'])
            start_response('200 OK', [])
            return []

        elif path == '/listdir':
            shard = urlparse.parse_qs(env['QUERY_STRING']).get('shard', None)
            if not shard:
                start_response('404 Not Found', [])
                return []
            attempts_cnt = urlparse.parse_qs(env['QUERY_STRING']).get('attempts_cnt', None)
            if attempts_cnt:
                attempts_cnt = int(attempts_cnt[0])

            shard = shard[0]  # only one shard is allowed
            observe_dir = os.path.join(os.path.realpath(targets_dir), shard)

            files = sorted(_listdir_recusive(observe_dir, dirs_limit=attempts_cnt))

            result = []
            for file_name in files:
                absolute_file_name = os.path.join(observe_dir, file_name)
                if os.path.isfile(absolute_file_name):
                    stat = os.stat(absolute_file_name)

                    result.append({
                        'name': file_name,
                        'size': stat.st_size,
                        'mtime': stat.st_mtime,
                    })

            start_response('200 OK', [('Content-Type', 'application/json')])
            return json.dumps(result, indent=4)

        elif path == '/getcontent':
            shard = urlparse.parse_qs(env['QUERY_STRING']).get('shard', None)
            if not shard:
                start_response('404 Not Found', [])
                return []

            shard = shard[0]  # only one shard is allowed
            observe_dir = os.path.join(os.path.realpath(targets_dir), shard)

            target_file_name = urlparse.parse_qs(env['QUERY_STRING']).get('file', None)
            if not target_file_name:
                start_response('404 Not Found', [])
                return []

            target_file_path = os.path.join(observe_dir, target_file_name[0])

            start_response('200 OK', [
                ('Content-Type', 'text/plain'),
                ('Content-Disposition', 'inline; filename={}'.format(os.path.basename(target_file_path))),
            ])
            with open(target_file_path) as f:
                return [f.read()]

        else:
            start_response('404 Not Found', [])
            return []

    logging.getLogger(__name__).info('Listening at %s', endpoint)
    WSGIServer(endpoint, serve_request, log=_LogAdapter(logging.getLogger(__name__), logging.DEBUG)).serve_forever()


# Gevent of version < 1.2 writes its log messages to file-like objects.
# I.e. using write(str). So the following wrapper is needed.
class _LogAdapter(object):
    def __init__(self, logger, level):
        self.logger = logger
        self.level = level

        self.ignore_messages = [
            '/admin?action=stat',
        ]

    def write(self, msg):
        for to_ignore in self.ignore_messages:
            if to_ignore in msg:
                return
        self.logger.log(self.level, msg)

    def flush(self):
        pass

    def writelines(self, lines):
        for line in lines:
            self.write(line)


def _listdir_recusive(path, dirs_limit=None):
    files = []

    dirs = os.listdir(path)
    if dirs_limit:
        dirs = sorted(dirs, reverse=True)[:dirs_limit]

    for dir_entity in dirs:
        if os.path.isdir(os.path.join(path, dir_entity)):
            for subdir_entity in _listdir_recusive(os.path.join(path, dir_entity)):
                files.append(os.path.join(dir_entity, subdir_entity))
        if os.path.isfile(os.path.join(path, dir_entity)):
            files.append(dir_entity)

    return files
