import datetime
import logging
import random
import pprint
from contextlib import contextmanager

from gevent import Greenlet, sleep
import retry

import infra.callisto.deploy.tracker.client as tracker
import bundle_build
import enums
import external
import utils
from storage import Storage


RECHECK_EACH = datetime.timedelta(hours=1)


class ShardTool2(object):
    def __init__(
            self, working_root, release_root, aux_working_root, max_dl_speed, max_ul_speed, copier_opts, logs_dir,
            max_inc_build_tries, recheck_shards=False,
    ):
        self.worker = Worker()
        self.shards_storage = Storage(working_root, release_root)
        self.aux_storage = Storage(aux_working_root, aux_working_root)

        self._loop_paused = False
        self._todo_shards = {}

        self._max_dl_speed = max_dl_speed
        self._max_ul_speed = max_ul_speed
        self._copier_opts = copier_opts
        self._logs_dir = logs_dir
        self._max_inc_build_tries = max_inc_build_tries
        self._enable_recheck_shards = recheck_shards

    # noinspection PyBroadException
    def run(self):
        while True:
            try:
                _log.debug('start iter')
                for shard in self._iter_buildable_shards():
                    self._build_shard(shard)

                if self._enable_recheck_shards:
                    self._recheck_shards()
            except Exception:
                logging.exception('Unhandled exception in the main loop')
            finally:
                _log.debug('end iter')
                sleep(5)

    def reconfigure(self, config):
        _log.debug('Reconfiguring FROM\n%s\n TO:\n%s', pprint.pformat(self.get_status()), pprint.pformat(config))

        with self._pause_loop():
            if self.worker.shard and self.worker.shard.name not in config:
                _log.info('Stop building [%s]', self.worker.shard)
                self.worker.stop()

            new_shards = {}
            needed_shards = set()
            needed_bundles = set()
            for shard in config:
                needed_shards.add(shard)

                if shard in self._todo_shards:
                    new_shards[shard] = self._todo_shards[shard]
                    if new_shards[shard].status != enums.Status.DONE:
                        new_shards[shard].config = config[shard]
                elif shard in self._ready_shards:
                    new_shards[shard] = Shard(shard, config[shard], enums.Status.DONE)
                else:
                    new_shards[shard] = Shard(shard, config[shard], enums.Status.IDLE)
                    self.shards_storage.rm_slot(shard)

                if (
                    new_shards[shard].prev_shard_name
                    and new_shards[shard].status != enums.Status.DONE
                    and new_shards[shard].attempt <= self._max_inc_build_tries
                ):
                    needed_shards.add(new_shards[shard].prev_shard_name)

                if new_shards[shard].bundle:
                    needed_bundles.add(new_shards[shard].bundle)

            self._cleanup(needed_shards, needed_bundles)
            self._todo_shards.clear()
            self._todo_shards = new_shards

    def get_status(self):
        result = {}
        for resource in self.shards_storage.ready_resources():
            result[resource.name] = {
                'method': '__unknown__',
                'status': enums.Status.DONE,
                'attempt': 0,
                'rbtorrent': resource.rbtorrent,
            }

        for shard in self._todo_shards.itervalues():
            if shard.name in result:
                continue

            result[shard.name] = {
                'method': shard.method,
                'status': shard.status,
                'attempt': shard.attempt,
            }

        return result

    def _cleanup(self, needed_shards, needed_bundles):
        self.shards_storage.cleanup(needed_shards)
        self.aux_storage.cleanup(needed_bundles)

    def _build_shard(self, shard):
        method = shard.config['method']
        method_fn = None
        if method == 'bundle_build':
            method_fn = self._bundle_build
        elif method == 'download':
            method_fn = self._download
        elif method == 'inc_build':
            method_fn = self._inc_build

        if method_fn:
            _log.info('Start building [%s] with %s', shard, method_fn)

            shard.status = enums.Status.BUILD
            shard.attempt += 1

            # TODO: collect worker exit status and set shard status accordingly
            self.worker = Worker(method_fn, shard)
            self.worker.run()  # killed greenlet exits silently
        else:
            _log.warning('Unknown build method: %s', method)

    def _bundle_build(self, shard):
        bundle_build.build(shard, self.shards_storage, self.aux_storage, self._logs_dir)

    def _download_torrent(self, rbtorrent, dst_dir, args):
        return external.sky_get(
            resource_url=rbtorrent,
            dst_dir=dst_dir,
            timeout=60000,
            max_dl_speed=args.get('max_dl_speed', self._max_dl_speed),
            max_ul_speed=args.get('max_ul_speed', self._max_ul_speed),
            copier_opts=args.get('copier_opts', self._copier_opts),
            hardlink=args.get('hardlinked', False),
        )

    def _inc_build(self, shard):
        shard.status = enums.Status.BUILD
        if not shard.prev_shard_name or shard.attempt > self._max_inc_build_tries:
            return self._bundle_build(shard)
        if shard.prev_shard_name not in self._ready_shards:
            rbtorrent = _resolve_prev_shard_rbtorrent(shard)
            if rbtorrent is None:
                return self._bundle_build(shard)
            recursive_resources = _resolve_prev_shard_recursive(shard)
            if recursive_resources:
                self._download_recursive(shard.prev_shard_name, rbtorrent, recursive_resources)
            else:
                self._download(Shard(
                    shard.prev_shard_name,
                    {'method': 'download', 'args': {'rbtorrent': rbtorrent}},
                    enums.Status.IDLE,
                ))
        if shard.prev_shard_name in self._ready_shards:
            bundle_build.build(
                shard,
                self.shards_storage,
                self.aux_storage,
                self._logs_dir,
                self.shards_storage.path_to_resource(shard.prev_shard_name),
            )
        else:
            shard.status = enums.Status.IDLE

    @property
    def _ready_shards(self):
        return [resource.name for resource in self.shards_storage.ready_resources()]

    @property
    def logs_dir(self):
        return self._logs_dir

    @contextmanager
    def _pause_loop(self):
        try:
            _log.debug('loop paused')
            self._loop_paused = True
            yield
        finally:
            _log.debug('loop resumed')
            self._loop_paused = False

    def _is_shard_buildable(self, shard_name):
        shard = self._todo_shards[shard_name]
        if shard.status == enums.Status.DONE:
            return False
        if (
            shard.status == enums.Status.FAILURE
            and shard.since_prev_failed_build() < datetime.timedelta(minutes=10)
        ):
            return False
        return True

    def _iter_buildable_shards(self):
        def shard_order(x, y):
            return cmp(x, y)

        for shard in utils.sort_and_iterate(self._todo_shards.keys(), cmp=shard_order):
            if self._loop_paused:
                break

            if self._is_shard_buildable(shard):
                yield self._todo_shards[shard]

    def _iter_shards_to_check(self):
        def shard_order(x, y):
            return cmp(y, x)

        for shard in utils.sort_and_iterate(self._todo_shards.keys(), cmp=shard_order):
            if self._loop_paused:
                break

            if (
                self._todo_shards[shard].status == enums.Status.DONE
                and self._todo_shards[shard].next_check < datetime.datetime.now()
            ):
                yield self._todo_shards[shard]

    def _download(self, shard):
        shard_dir = self.shards_storage.get_slot(shard.name, build=False)
        args = shard.config['args']

        rbtorrent = args.get('rbtorrent') or external.find_rbtorrent(shard.name)
        if not rbtorrent:
            shard.status = enums.Status.IDLE

        elif self._download_torrent(rbtorrent, shard_dir, args):
            self.shards_storage.set_ready(shard.name, build=False, rbtorrent=rbtorrent)
            shard.status = enums.Status.DONE
            shard.set_checked()
        else:
            shard.status = enums.Status.IDLE

    def _download_recursive(self, shard_name, rbtorrent, resources):
        shard_dir = self.shards_storage.get_slot(shard_name, build=False)
        tracker.download_recursive(shard_dir, resources, resource_attempts_count=3)
        _log.debug('downloaded recursive')
        new_rbtorrent = external.sky_share(shard_dir)
        if rbtorrent != new_rbtorrent:
            _log.warning('recursive rbtorrent not equal to full %s != %s', rbtorrent, new_rbtorrent)
        else:
            _log.info('successfully downloaded %s', shard_name)
            self.shards_storage.set_ready(shard_name, build=False, rbtorrent=rbtorrent)

    def _recheck_shards(self):
        for shard in self._iter_shards_to_check():
            _log.info('recheck %s', shard.name)
            rbtorrent_in_storage = self.shards_storage.read_rbtorrent(shard.name)
            current_rbtorrent = external.sky_share(self.shards_storage.path_to_resource(shard.name))
            if rbtorrent_in_storage != current_rbtorrent:
                _log.warning(
                    '%s rbtorrent differs: was %s != new %s',
                    shard.name, rbtorrent_in_storage, current_rbtorrent,
                )
                self.shards_storage.rm_slot(shard.name)
                shard.status = enums.Status.IDLE
            else:
                _log.info('recheck %s ok', shard.name)
                shard.set_checked()


class Worker(object):
    def __init__(self, fn=None, shard=None):
        self.greenlet = Greenlet(fn, shard) if fn else None
        self.shard = shard

    def run(self):
        if self.greenlet is not None:
            self.greenlet.start()
            self.greenlet.join()

    def stop(self):
        if self.greenlet is not None:
            self.greenlet.kill()


class Shard(object):
    next_check = datetime.datetime.max
    last_fail_time = datetime.datetime.max

    def __init__(self, name, config, status):
        self.name = name
        self.config = config
        self.status = status
        self.attempt = 0
        self.set_checked()

    def set_checked(self):
        delta = RECHECK_EACH + datetime.timedelta(minutes=random.randint(0, 60))
        self.next_check = datetime.datetime.now() + delta

    def on_failed_build(self):
        self.last_fail_time = datetime.datetime.now()

    def since_prev_failed_build(self):
        return datetime.datetime.now() - self.last_fail_time

    @property
    def method(self):
        return self.config.get('method', '__unknown__')

    @property
    def prev_shard_name(self):
        if 'prev_shard' in self.config:
            return self.config['prev_shard']['name']
        if 'prev_shard_name' in self.config:
            return self.config['prev_shard_name']
        return None

    @property
    def bundle(self):
        return self.config.get('args', {}).get('bundle', {}).get('rbtorrent')

    def __str__(self):
        return '<shard [{}], method [{}], status [{}]>'.format(self.name, self.method, self.status)


@retry.retry(tries=5, delay=5)
def _resolve_prev_shard_rbtorrent(shard):
    if 'prev_shard' in shard.config:
        prev_shard = shard.config['prev_shard']
        tracker_client = tracker.Client(prev_shard['tracker_url'])
        try:
            return tracker_client.resolve_one(prev_shard['namespace'], prev_shard['name']).rbtorrent
        except tracker.ResourceNotFound:
            return None
    elif 'prev_shard_name' in shard.config:
        return external.find_rbtorrent(shard.config['prev_shard_name'])
    return None


@retry.retry(tries=5, delay=30, backoff=2)
def _resolve_prev_shard_recursive(shard):
    if 'prev_shard' in shard.config:
        prev_shard = shard.config['prev_shard']
        tracker_client = tracker.Client(prev_shard['tracker_url'])
        if prev_shard.get('recursive_download', False):
            return tracker_client.match_resources(
                prev_shard['namespace'],
                prev_shard.get('regexp') or prev_shard['name'] + '/',
            )
    return None


_log = logging.getLogger(__name__)
