import contextlib
import logging
import os
import requests.exceptions
import subprocess
import time

import porto
import retry

import storage
import utils


def shoot(storage, shard, basesearch_port, results_dir, threads_count, mem_limit_gb):
    dexecutor = storage.get_resource('da379e254bdfef40985ddf88926e41c5cf586f82') + '/d-executor'
    _log.debug('with shooter [%s]', dexecutor)

    ddumper = storage.get_resource('b10de8fc609682f92f19a29a83e935c785fde6a7') + '/d-dumper'
    _log.debug('with dumper [%s]', ddumper)

    plan = storage.get_resource('eba5a827d6ed4d97ecd24498ee207821da4265c8') + '/grelka.plan'
    _log.debug('with plan [%s]', plan)

    with _parent_container():
        # basesearch_container = _subcontainer('basesearch', threads_count, mem_limit_gb)

        for attempt in range(5):
            raw_results = os.path.join(results_dir, 'raw_results.thr.{}'.format(attempt))
            utils.remove_file(raw_results)
            dexecutor_container = _subcontainer('dexecutor', threads_count=2, mem_limit_gb=5)
            _run_dexecutor(dexecutor_container, basesearch_port, dexecutor, plan, raw_results, threads_count)

            result_numbers = os.path.join(results_dir, 'result_numbers.thr.{}'.format(attempt))
            utils.remove_file(result_numbers)
            _run_ddumper(ddumper, raw_results, result_numbers)

            dexecutor_container.Destroy()


@contextlib.contextmanager
def _parent_container(name='shoot_basesearch'):
    container = None
    connection = None

    try:
        connection = porto.Connection()
        connection.connect()

        container = connection.Create(name)

        yield container

    finally:
        if container:
            container.Destroy()
        if connection:
            connection.disconnect()


def _subcontainer(name, threads_count, mem_limit_gb):
    connection = porto.Connection()
    container = connection.Create('shoot_basesearch/' + name)

    container.SetProperty('memory_guarantee', '{}G'.format(mem_limit_gb - 1))
    container.SetProperty('memory_limit', '{}G'.format(mem_limit_gb))
    container.SetProperty('recharge_on_pgfault', True)
    container.SetProperty('cpu_limit', '{}c'.format(threads_count))
    container.SetProperty('cpu_guarantee', '{}c'.format(threads_count))
    container.SetProperty('cpu_set', 'threads {}'.format(threads_count))

    return container


def _start_basesearch(container, shard, basesearch_port):
    basesearch = './basesearch'
    _log.debug('with basesearch [%s]', basesearch)

    config = storage.get_resource('c1af95c840e249e87009ceedf1e92eca0cda9ed2') + '/config'
    _log.debug('with config [%s]', config)

    models = storage.get_resource('66f57b77713760272f705b21ac6efb7eca8a8b92') + '/models.archive'
    _log.debug('with models [%s]', models)

    _log.debug('with shard [%s]', shard)

    container.SetProperty('command', _get_basesearch_cmd(basesearch, basesearch_port, shard, config, models))
    container.Start()

    time.sleep(2)

    _wait_for_basesearch_start(basesearch_port)


def _run_dexecutor(container, basesearch_port, dexecutor, plan, raw_results, threads_count):
    command = ' '.join([
        dexecutor,
        '--mode', 'finger',
        '--plan-file', plan,
        '--simultaneous', str(threads_count),
        '--replace-host', 'localhost',
        '--replace-port', str(basesearch_port),
        '--output', raw_results,
        '--queries-limit', '50000',
        '--augmenturl', '"&pron=ignore_db_timestamp"',
    ])
    container.SetProperty('command', command)
    container.Start()
    time.sleep(1)
    container.Wait()


def _run_ddumper(ddumper, raw_results, result_numbers):
    with open(result_numbers, 'w') as output:
        subprocess.call([
            ddumper,
            '--timeline',
            '--input', raw_results,
        ], stdout=output)


@retry.retry(requests.exceptions.ConnectionError, tries=20, delay=1)
def _wait_for_basesearch_start(basesearch_port):
    _log.debug(requests.get('http://localhost:{}/yandsearch?info=getconfig'.format(basesearch_port)).text)
    _log.info('basesearch started')


def _get_basesearch_cmd(basesearch, basesearch_port, shard, config, models):
    return '{basesearch} -p {port} -d {config} -V MXNetFile={models} -V IndexDir={shard}'.format(
        basesearch=basesearch,
        port=basesearch_port,
        config=config,
        shard=shard,
        models=models,
    ) + ' -V LoadLog=/dev/null -V PassageLog=/dev/null'


def _start2(root):
    connection = porto.Connection()
    container = connection.CreateWeakContainer('shoot_basesearch')
    container.SetProperty('cpu_guarantee', '8c')
    container.SetProperty('cpu_limit', '8c')
    container.SetProperty('cwd', root)
    container.SetProperty('command',
                          _get_basesearch_cmd(
                              root + '/basesearch',
                              33221,
                              root + '/primus-WebTier0-0-24-1512858920',
                              root + '/basesearch.cfg',
                              root + '/models.archive',
                          ))
    _log.info('Starting {}', container.GetProperty('command'))
    container.Start()
    container.Wait()


_log = logging.getLogger(__name__)


def main():
    logging.basicConfig(level=logging.DEBUG)
    _start2('/ssd/okats/storage_root')
