import os
import logging
import collections

import external
import utils


class Storage(object):
    def __init__(self, working_root, release_root):
        self._working_root = os.path.abspath(working_root)
        self._release_root = os.path.abspath(release_root)
        utils.ensure_dir(os.path.join(self._release_root))
        utils.ensure_dir(os.path.join(self._working_root, 'build'))
        utils.ensure_dir(os.path.join(self._working_root, 'download'))
        utils.ensure_dir(os.path.join(self._working_root, 'ready'))

    @property
    def used(self):
        return utils.du(self._working_root, exclude_dotfiles=False)

    @property
    def free(self):
        return utils.free_space(self._working_root)

    @property
    def resources(self):
        return set(
            os.listdir(os.path.join(self._working_root, 'build')) +
            os.listdir(os.path.join(self._working_root, 'download'))
        )

    def read_rbtorrent(self, resource):
        return _read_rbtorrent(self._ready_path(resource))

    def ready_resources(self):
        for resource in self.resources:
            rbtorrent = _read_rbtorrent(self._ready_path(resource))
            if rbtorrent:
                yield Resource(name=resource, ready=True, rbtorrent=rbtorrent)

    def get_slot(self, resource_id, build=False):
        path = self._slot_path(resource_id, build)
        utils.ensure_dir(path)
        return path

    def rm_slot(self, resource):
        _log.info('Removing [%s]', resource)

        utils.remove_dir(self._slot_path(resource, build=True))
        utils.remove_dir(self._slot_path(resource, build=False))
        utils.remove_file(self._ready_path(resource))
        utils.remove_file(self._ready_path(resource) + '.rbtorrent')

    def share(self, resource, build=True):
        _log.info('Sharing [%s]', resource)
        rbtorrent = external.sky_share(self._slot_path(resource, build))
        return rbtorrent

    def set_ready(self, resource, build, rbtorrent):
        _log.info('Releasing [%s]', resource)
        assert rbtorrent

        utils.remove_file(self._ready_path(resource))
        utils.create_symlink(self._slot_path(resource, build=build), self._ready_path(resource))
        utils.remove_file(self._release_path(resource))
        utils.create_symlink(self._slot_path(resource, build=build), self._release_path(resource))

        _write_rbtorrent(self._ready_path(resource), rbtorrent)

    def get_resource(self, rbtorrent, **kwargs):
        download_dir = self.get_slot(rbtorrent, build=False)

        if not external.sky_get(
            resource_url=rbtorrent,
            dst_dir=download_dir,
            timeout=60000,
            **kwargs
        ):
            raise RuntimeError('Unable to download %s', rbtorrent)

        self.set_ready(rbtorrent, build=False, rbtorrent=rbtorrent)

        return download_dir

    def _slot_path(self, resource, build):
        return os.path.join(self._working_root, 'build' if build else 'download', resource)

    def _ready_path(self, resource):
        return os.path.join(self._working_root, 'ready', resource)

    def _release_path(self, resource):
        return os.path.join(self._release_root, resource)

    def path_to_resource(self, resource):
        return self._ready_path(resource)

    def cleanup(self, needed_resources):
        for resource in self.resources:
            if resource not in needed_resources:
                self.rm_slot(resource)


Resource = collections.namedtuple('Resource', ['name', 'ready', 'rbtorrent'])


def _write_rbtorrent(shard_dir, rbtorrent):
    with open(shard_dir + '.rbtorrent', 'w') as f:
        _log.debug('Write rbtorrent [%s]', rbtorrent)
        f.write(rbtorrent + '\n')


def _read_rbtorrent(shard_dir):
    # noinspection PyBroadException
    try:
        with open(shard_dir + '.rbtorrent', 'r') as f:
            rbtorrent = f.read().strip()
            _log.debug('Read rbtorrent [%s]', rbtorrent)
            return rbtorrent
    except Exception:
        return None


_log = logging.getLogger(__name__)


def test_operations():
    map(utils.recreate_dir, [
        './storage_root',
        './storage_root/build',
        './storage_root/build/p1',
        './storage_root/build/p2',
        './storage_root/build/p3',
        './storage_root/download',
        './storage_root/download/p1',
        './storage_root/download/p5',
        './storage_root/ready',
        './storage_root/ready/r1',
        './storage_root/ready/r2',
        './storage_root/ready/r5',
        './storage_root/p1',
        './storage_root/p5',
        './storage_root/p10',
    ])
    storage = Storage('./storage_root', './release_root')

    slot = storage.get_slot('resource-1', build=True)
    with open(slot + '/data', 'a') as f:
        f.write('a' * 1024 * 1024)
    assert 1024 * 1024 < storage.used < 1.2 * 1024 * 1024

    storage.rm_slot('resource-1')
    assert storage.used < 1024 * 1024

    storage.get_slot('resource-2', build=False)
    assert {'p1', 'p2', 'p3', 'p5', 'resource-2'} == storage.resources
    storage.rm_slot('resource-2')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources
    storage.rm_slot('resource-2')
    storage.rm_slot('resource-2')
    storage.rm_slot('resource-2')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources

    slot = storage.get_slot('resource-3', build=True)
    with open(slot + '/data', 'a') as f:
        f.write('a' * 1024)
    storage.share('resource-3')
    storage.set_ready('resource-3', build=True, rbtorrent='fake_rb_torrent')
    storage.rm_slot('resource-3')
    assert {'p1', 'p2', 'p3', 'p5'} == storage.resources
