import datetime
import gevent
import gevent.select
import logging
import os
import json
import shutil
import subprocess

import retry


def recreate_dir(path):
    remove_dir(path)
    os.makedirs(path)


def remove_dir(path):
    if os.path.exists(path):
        if os.path.isdir(path):
            shutil.rmtree(path)
        elif os.path.isfile(path):
            os.remove(path)


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)


def create_symlink(src, dst):
    os.symlink(src, dst)


def remove_file(path):
    try:
        os.remove(path)
    except OSError:
        pass


def ls(path):
    return [os.path.join(path, d) for d in os.listdir(path)]


def free_space(path):
    s = os.statvfs(path)
    return s.f_frsize * s.f_bavail


def du(path, exclude_dotfiles=True):
    p = subprocess.Popen(
        'du -bs {} {}'.format('--exclude ".tmp"' if exclude_dotfiles else '', path),
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        shell=True,
    )
    out, err = p.communicate()

    if p.returncode:
        raise RuntimeError('du failed with code {}, stderr: [{}]'.format(p.returncode, err))

    return int(out.split()[0])


def file_last_access(path):
    return datetime.datetime.fromtimestamp(int(os.path.getmtime(path)))


def loop_forever(fn, *args, **kwargs):
    while True:
        # noinspection PyBroadException
        try:
            fn(*args, **kwargs)
        except Exception:
            logging.exception('Unhandled exception in the loop <%s>', fn)

        gevent.sleep(1)


def prepare_env(upd):
    env = os.environ.copy()
    for key, value in upd.iteritems():
        env[str(key)] = str(value).lower()
    return env


def sort_and_iterate(iterable, cmp=cmp):
    sorted_items = sorted(iterable, cmp=cmp)
    for item in sorted_items:
        if sorted_items != sorted(iterable, cmp=cmp):
            raise RuntimeError('iterable changed during iteration')
        yield item


@retry.retry(exceptions=[gevent.Timeout], tries=3, delay=10)
def sky_files(rbtorrent):
    popen = subprocess.Popen(['sky', 'files', '--json', rbtorrent], stdout=subprocess.PIPE)
    try:
        with gevent.Timeout(seconds=45):
            popen.wait()
            assert popen.returncode == 0
            files = json.loads(popen.stdout.read())
            files.sort(key=lambda x: x['name'])
            return files
    except Exception:
        popen.kill()
        raise


def copy_file(src, dst):
    shutil.copyfile(src, dst)


def walk_dirs(directory, max_depth=10):
    if max_depth == 0:
        return []

    result = []
    for sub_dir in os.listdir(directory):
        sub_dir_full_path = os.path.join(directory, sub_dir)
        if os.path.isdir(sub_dir_full_path):
            result += walk_dirs(sub_dir_full_path, max_depth - 1)

    return result + [directory]
