import cgi

import json
import gevent
import requests
import telegram
import telegram.ext

import infra.callisto.controllers.sdk.notify as notify

import utils
import context
import projects


class Command(object):
    name = ''
    description = ''

    def reply_message(self, bot, update):
        raise NotImplementedError

    def register(self, dp):
        dp.add_handler(telegram.ext.CommandHandler(self.name, self.reply_message))
        if hasattr(self, 'callback'):
            dp.add_handler(telegram.ext.CallbackQueryHandler(self.callback, pattern='^{}-'.format(self.name)))

    def wrap_callback_data(self, text):
        return '{}-{}'.format(self.name, text)

    def unwrap_callback_data(self, text):
        return text.replace('{}-'.format(self.name), '')


class StartCommand(Command):
    name = 'start'
    description = 'enable me'

    def reply_message(self, bot, update):
        settings = context.settings_storage().load_settings(update.message.chat_id)
        settings.enabled = True
        context.settings_storage().store_settings(update.message.chat_id, settings)

        update.message.reply_text('Now I am enabled')


class StopCommand(Command):
    name = 'stop'
    description = 'disable me'

    def reply_message(self, bot, update):
        settings = context.settings_storage().load_settings(update.message.chat_id)
        settings.enabled = False
        context.settings_storage().store_settings(update.message.chat_id, settings)

        update.message.reply_text('OK. You can enable me with /start command')


class SubscribeCommand(Command):
    name = 'subscribe'
    description = 'subscribe to projects notifications'

    def _make_keyboard(self, settings):
        def str_prj(prj):
            return '{} - {}'.format(prj.name, 'on' if prj in settings.projects else 'off')

        button_list = [
            telegram.InlineKeyboardButton(str_prj(project), callback_data=self.wrap_callback_data(project.name))
            for project in projects.list_projects()
        ]
        return telegram.InlineKeyboardMarkup(utils.build_menu(button_list, n_cols=3))

    def reply_message(self, bot, update):
        settings = context.settings_storage().load_settings(update.message.chat_id)
        bot.send_message(update.message.chat_id, 'Select projects', reply_markup=self._make_keyboard(settings))

    def callback(self, bot, update):
        query = update.callback_query

        project = projects.get_project(self.unwrap_callback_data(query.data))
        settings = context.settings_storage().load_settings(query.message.chat_id)
        settings.projects ^= {project}
        context.settings_storage().store_settings(query.message.chat_id, settings)

        bot.edit_message_reply_markup(
            chat_id=query.message.chat_id,
            message_id=query.message.message_id,
            reply_markup=self._make_keyboard(settings),
        )


class _DumpCommand(Command):
    level = ''

    def _make_keyboard(self):
        button_list = [
            telegram.InlineKeyboardButton(source.name, callback_data=self.wrap_callback_data(source.name))
            for source in projects.list_sources()
        ]
        return telegram.InlineKeyboardMarkup(utils.build_menu(button_list, n_cols=3))

    def reply_message(self, bot, update):
        bot.send_message(update.message.chat_id, 'Select ctrl', reply_markup=self._make_keyboard())

    def callback(self, bot, update):
        source = projects.get_source(self.unwrap_callback_data(update.callback_query.data))
        notifications = context.monitor().source_notifications(source, self.level)
        if notifications:
            utils.send_message(
                bot,
                update.callback_query.message.chat_id,
                utils.source_notifications_to_text(source, notifications)
            )
        else:
            bot.send_message(update.callback_query.message.chat_id, 'no notifications for {}'.format(source.name))


class DumpCommand(_DumpCommand):
    name = 'dump'
    description = 'dump important notifications'
    level = notify.NotifyLevels.INFO


class DumpAllCommand(_DumpCommand):
    name = 'dump_all'
    description = 'dump all notifications'
    level = notify.NotifyLevels.IDLE


class TraceBackCommand(Command):
    name = 'traceback'
    description = 'recent traceback'

    @staticmethod
    def load_traceback(url):
        try:
            resp = requests.get(url).json()
            iters = resp['failed'] + resp['succeed']
            iters.sort(key=lambda x: x['started'])
            if iters:
                return '\n'.join([
                    'Started: {}'.format(iters[-1]['started']),
                    'Finished: {}'.format(iters[-1]['finished']),
                    'Traceback: {}'.format(iters[-1].get('traceback', 'No traceback'))
                ])
            else:
                return 'no iterations yet'
        except (requests.RequestException, ValueError) as exc:
            return 'Failed: {}'.format(exc)

    def _make_keyboard(self):
        button_list = [
            telegram.InlineKeyboardButton(source.name, callback_data=self.wrap_callback_data(source.name))
            for source in projects.list_sources()
        ]
        return telegram.InlineKeyboardMarkup(utils.build_menu(button_list, n_cols=3))

    def reply_message(self, bot, update):
        bot.send_message(update.message.chat_id, 'Select source', reply_markup=self._make_keyboard())

    def callback(self, bot, update):
        source = projects.get_source(self.unwrap_callback_data(update.callback_query.data))
        text = self.load_traceback(source.ctrl_url + 'info/timeline')
        text = cgi.escape(text)
        href = '<a href="{}">{}</a>'.format(source.ctrl_url, source.name)
        text = 'Recent iteration of {}:\n{}'.format(href, text)
        utils.send_message(bot, update.callback_query.message.chat_id, text)


class StatusCommand(Command):
    name = 'status'
    description = 'ctrl status'

    @staticmethod
    def load_status(url):
        try:
            resp = requests.get(url).json()
            return json.dumps(resp, indent=2)
        except (requests.RequestException, ValueError) as exc:
            return 'Failed: {}'.format(exc)

    def _make_keyboard(self):
        button_list = [
            telegram.InlineKeyboardButton(source.name, callback_data=self.wrap_callback_data(source.name))
            for source in projects.list_sources()
        ]
        return telegram.InlineKeyboardMarkup(utils.build_menu(button_list, n_cols=3))

    def reply_message(self, bot, update):
        bot.send_message(update.message.chat_id, 'Select source', reply_markup=self._make_keyboard())

    def callback(self, bot, update):
        source = projects.get_source(self.unwrap_callback_data(update.callback_query.data))
        text = self.load_status(source.ctrl_url + 'status')
        text = cgi.escape(text)
        href = '<a href="{}">{}</a>'.format(source.ctrl_url, source.name)
        text = 'Status of {}:\n{}'.format(href, text)
        utils.send_message(bot, update.callback_query.message.chat_id, text)


class HelpCommand(Command):
    name = 'help'
    description = 'show this help'

    def reply_message(self, bot, update):
        commands = ['List of commands:'] + ['/{} - {}'.format(cmd.name, cmd.description) for cmd in COMMANDS]
        update.message.reply_text('\n'.join(commands))


COMMANDS = [
    StartCommand(),
    StopCommand(),
    SubscribeCommand(),
    DumpCommand(),
    DumpAllCommand(),
    TraceBackCommand(),
    StatusCommand(),
    HelpCommand(),
]


class ConversationHandler(object):
    def __init__(self, token):
        self._updater = telegram.ext.Updater(token=token)
        dp = self._updater.dispatcher
        for cmd in COMMANDS:
            cmd.register(dp)

    def start(self):
        gevent.spawn(self._updater.start_polling)
