import logging

import gevent
import requests

import infra.callisto.controllers.sdk.notify as notify

import projects


class Notification(object):
    def __init__(self, prj, source, message, level):
        self.prj = prj
        self.source = source
        self.message = message
        self.level = level


class SourceObserver(object):
    def __init__(self, project, source):
        self._project = project
        self._source = source
        self._state = []

    def update(self):
        try:
            notifications = _load_notifications(
                self._source.notifications_url,
                self._project.name,
                self._source.name,
            )
            if notifications is not None:
                self._state = notifications
        except Exception:
            logging.exception('')

    def notifications(self, level):
        return sorted((
            notification for notification in self._state
            if notification.level >= level
        ), key=lambda x: x.level, reverse=True)


class ProjectObserver(object):
    def __init__(self, project):
        self._project = project
        self._observers = {source: SourceObserver(self._project, source) for source in self._project.sources}

    def update(self):
        for observer in self._observers.values():
            observer.update()

    def project_notifications(self, level):
        res = {}
        for source, observer in self._observers.items():
            res[source] = observer.notifications(level)
        return res

    def source_notifications(self, source, level):
        return self._observers[source].notifications(level)


class Monitor(object):
    def __init__(self):
        self._observers = {prj: ProjectObserver(prj) for prj in projects.PROJECTS.values()}

    def start(self):
        gevent.spawn(self._eternal_update)

    def _eternal_update(self):
        while True:
            logging.debug('monitor iter')
            for observer in self._observers.values():
                observer.update()
            gevent.sleep(15)

    def project_notifications(self, project, level):
        return self._observers[project].project_notifications(level)

    def source_notifications(self, source, level):
        return self._observers[source.project].source_notifications(source, level)


def _load_notifications(url, prj_name, source_name):
    try:
        resp = requests.get(url, timeout=10)
        resp.raise_for_status()
        return [
            Notification(prj_name, source_name, notification['message'], _level_from_str(notification['level']))
            for notification in resp.json()
        ]
    except requests.RequestException:
        return None


def _level_from_str(string):
    for level in notify.NotifyLevels.ALL:
        if str(level) == string:
            return level
    return notify.NotifyLevels.IDLE
