import os
import json
import tempfile
import datetime

import infra.callisto.controllers.sdk.notify as notify

import projects as projects


class ChatSettings(object):
    def __init__(
            self,
            enabled=False,
            project_names=(),
            notify_each_seconds=15 * 60,
    ):
        self.enabled = enabled
        self.projects = self._load_projects(project_names)
        self.notify_each_seconds = int(notify_each_seconds)

    @classmethod
    def current_level(cls):
        if 0 <= datetime.datetime.now().hour <= 8:
            return notify.NotifyLevels.ERROR
        else:
            return notify.NotifyLevels.WARNING

    def dump_json(self):
        return {
            'enabled': self.enabled,
            'project_names': self._dump_projects(self.projects),
            'notify_each_seconds': self.notify_each_seconds,
        }

    @classmethod
    def _dump_projects(cls, projects_):
        return [project.name for project in projects_]

    @classmethod
    def _load_projects(cls, project_names):
        return {projects.get_project(name) for name in project_names}


class SettingsStorage(object):
    def __init__(self):
        self._file_path = './settings_dump.json'
        self._load()

    def store_settings(self, chat_id, settings):
        self._chats[chat_id] = settings
        self._dump()

    def load_settings(self, chat_id):
        return self._chats.get(chat_id, ChatSettings())

    def list_chats(self):
        return self._chats.keys()

    def _dump(self):
        data = {
            str(chat_id): settings.dump_json()
            for chat_id, settings in self._chats.items()
        }
        with tempfile.NamedTemporaryFile(dir='./', delete=False) as fp:
            json.dump(data, fp)
            fp.close()
            os.rename(fp.name, self._file_path)

    def _load(self):
        data = {}
        if os.path.exists(self._file_path):
            with open(self._file_path) as fp:
                data = json.load(fp)
        self._chats = {int(chat_id): ChatSettings(**settings) for chat_id, settings in data.items()}
