import infra.callisto.controllers.utils.gencfg_api as gencfg_api
import infra.callisto.controllers.utils.yp_utils as yp_utils

import argparse
import collections
import logging
import pprint

GROUPS_PODSETS = {
    # https://nanny.yandex-team.ru/ui/#/services/catalog/yp-export-sas/files#group_mapping.json
    'sas': {
        ('SAS_CALLISTO_PUMPKIN_BUILD', ): None,
        ('SAS_DISK_LUCENE', ): ('disk-search-backend-yp-prod', ),
        ('SAS_IMGS_BASE', ): ('sas-images-search.sas-images-tier0-base', ),
        ('SAS_IMGS_INVERTED_INDEX', ): ('sas-images-search.inverted-index', ),
        ('SAS_IMGS_RIM_3K', ): ('sas-imgs-rim.sas-imgs-rim', ),
        ('SAS_IMGS_T1_BASE', ): ('sas-images-search.sas-images-tier1-base', ),
        ('SAS_MAIL_LUCENE', ): ('mail-search-backend-prod', ),
        ('SAS_VIDEO_PLATINUM_INT', ): None,
        ('SAS_VIDEO_PLATINUM_INT_HAMSTER', ): None,
        ('SAS_VIDEO_TIER0_INT', ): None,
        ('SAS_VIDEO_TIER0_INT_HAMSTER', ): None,
        ('SAS_VIDEO_PLATINUM_BASE', ): ('sas-video-search.sas-video-platinum-base', ),
        ('SAS_VIDEO_PLATINUM_EMBEDDING', ): ('sas-video-search.platinum-embedding', ),
        ('SAS_VIDEO_PLATINUM_INVERTED_INDEX', ): ('sas-video-search.platinum-inverted-index', ),
        ('SAS_VIDEO_TIER0_BASE', ): ('sas-video-search.sas-video-tier0-base', ),
        ('SAS_VIDEO_TIER0_EMBEDDING', ): ('sas-video-search.tier0-embedding', ),
        ('SAS_VIDEO_TIER0_INVERTED_INDEX', ): ('sas-video-search.tier0-inverted-index', ),
        ('SAS_WEB_TIER0_ATTRIBUTE_BASE', ): ('sas-web-search.tier0-attr-base', ),
        ('SAS_WEB_TIER0_BASE', ): ('sas-web-search.tier0-base', ),
        ('SAS_WEB_TIER0_EMBEDDING', ): ('sas-web-search.tier0-embedding', ),
        ('SAS_WEB_TIER0_INT', ): ('sas-web-search.tier0-int', ),
        ('SAS_WEB_TIER0_INTL2', ): ('sas-web-search.tier0-intl2', ),
        ('SAS_WEB_TIER0_INVERTED_INDEX', ): ('sas-web-search.tier0-inverted-index', ),
        ('SAS_WEB_TIER0_KEYINV', ): ('sas-web-search.tier0-keyinv', ),
        ('SAS_WEB_TIER0_INTL2_HAMSTER', ): None,
        ('SAS_WEB_TIER0_INTL2_MULTI', ): None,
        ('SAS_WEB_TIER0_INT_HAMSTER', ): None,
        ('SAS_WEB_TIER0_INT_MULTI', ): None,
    },

    # https://nanny.yandex-team.ru/ui/#/services/catalog/yp-export-man/files#group_mapping.json
    'man': {
        ('MAN_DISK_LUCENE', ): ('disk-search-backend-yp-prod', ),
        ('MAN_IMGS_BASE', ): ('man-images-search.man-images-tier0-base', ),
        ('MAN_IMGS_RIM_3K', ): ('man-imgs-rim.man-imgs-rim', ),
        ('MAN_IMGS_T1_BASE', ): ('man-images-search.man-images-tier1-base', ),
        ('MAN_MAIL_LUCENE', ): ('mail-search-backend-prod', ),
        ('MAN_VIDEO_PLATINUM_BASE', ): ('man-video-search.man-video-platinum-base', ),
        ('MAN_VIDEO_PLATINUM_EMBEDDING', ): ('man-video-search.platinum-embedding', ),
        ('MAN_VIDEO_PLATINUM_INVERTED_INDEX', ): ('man-video-search.platinum-inverted-index', ),
        ('MAN_VIDEO_TIER0_BASE', ): ('man-video-search.man-video-tier0-base', ),
        ('MAN_VIDEO_TIER0_EMBEDDING', ): ('man-video-search.tier0-embedding', ),
        ('MAN_VIDEO_TIER0_INVERTED_INDEX', ): ('man-video-search.tier0-inverted-index', ),
        ('MAN_WEB_PLATINUM_JUPITER_BASE', ): ('man-web-search.man-web-platinum-base', ),
        ('MAN_WEB_PLATINUM_JUPITER_INT', ):  ('man-web-search.platinum-int',),
        ('MAN_WEB_PLATINUM_JUPITER_INT_HAMSTER', ):  ('man-web-search.platinum-int-hamster',),
        ('MAN_WEB_PLATINUM_JUPITER_INT_MULTI', ):  ('man-web-search.platinum-int-multi',),
        ('MAN_WEB_TIER0_ATTRIBUTE_BASE', ):  ('man-web-search.tier0-attr-base',),
        ('MAN_WEB_TIER0_BASE', ):  ('man-web-search.tier0-base',),
        ('MAN_WEB_TIER0_EMBEDDING', ):  ('man-web-search.tier0-embedding',),
        ('MAN_WEB_TIER0_INT', ):  ('man-web-search.tier0-int',),
        ('MAN_WEB_TIER0_INTL2', ):  ('man-web-search.tier0-intl2',),
        ('MAN_WEB_TIER0_INTL2_HAMSTER', ):  ('man-web-search.tier0-intl2-hamster',),
        ('MAN_WEB_TIER0_INTL2_MULTI', ):  ('man-web-search.tier0-intl2-multi',),
        ('MAN_WEB_TIER0_INT_HAMSTER', ):  ('man-web-search.tier0-int-hamster',),
        ('MAN_WEB_TIER0_INT_MULTI', ):  ('man-web-search.tier0-int-multi',),
        ('MAN_WEB_TIER0_INVERTED_INDEX', ):  ('man-web-search.tier0-inverted-index',),
        ('MAN_WEB_TIER0_KEYINV', ):  ('man-web-search.tier0-keyinv',),
    },

    # https://nanny.yandex-team.ru/ui/#/services/catalog/yp-export-vla/files#group_mapping.json
    'vla': {
        (
            'ALL_DISK_LUCENE_EXTRA_REPLICA',
            'VLA_DISK_LUCENE'
        ): ('disk-search-backend-yp-prod', ),
        (
            'ALL_MAIL_LUCENE_EXTRA_REPLICA',
            'VLA_MAIL_LUCENE'
        ): ('mail-search-backend-prod', ),
        ('VLA_IMGS_BASE', ): ('vla-images-search.vla-images-tier0-base', ),
        ('VLA_IMGS_T1_BASE', ): ('vla-images-search.vla-images-tier1-base', ),
        ('VLA_VIDEO_PLATINUM_BASE', ): ('vla-video-search.vla-video-platinum-base', ),
        ('VLA_VIDEO_PLATINUM_EMBEDDING', ): ('vla-video-search.platinum-embedding', ),
        ('VLA_VIDEO_PLATINUM_INVERTED_INDEX', ): ('vla-video-search.platinum-inverted-index', ),
        ('VLA_VIDEO_TIER0_BASE', ): ('vla-video-search.vla-video-tier0-base', ),
        ('VLA_VIDEO_TIER0_EMBEDDING', ): ('vla-video-search.tier0-embedding', ),
        ('VLA_VIDEO_TIER0_INVERTED_INDEX', ): ('vla-video-search.tier0-inverted-index', ),
        ('VLA_WEB_PLATINUM_JUPITER_BASE', ): ('vla-web-search.vla-web-platinum-base', ),
        ('VLA_WEB_PLATINUM_JUPITER_INT', ): ('vla-web-search.platinum-int', ),
        ('VLA_WEB_PLATINUM_JUPITER_INT_HAMSTER', ): None,
        ('VLA_WEB_PLATINUM_JUPITER_INT_MULTI', ): None,
        ('VLA_WEB_TIER0_ATTRIBUTE_BASE', ): ('vla-web-search.tier0-attr-base', ),
        ('VLA_WEB_TIER0_BASE', ): ('vla-web-search.tier0-base', ),
        ('VLA_WEB_TIER0_EMBEDDING', ): ('vla-web-search.tier0-embedding', ),
        ('VLA_WEB_TIER0_INT', ): ('vla-web-search.tier0-int', ),
        ('VLA_WEB_TIER0_INTL2', ): ('vla-web-search.tier0-intl2', ),
        ('VLA_WEB_TIER0_INTL2_HAMSTER', ): None,
        ('VLA_WEB_TIER0_INTL2_MULTI', ): None,
        ('VLA_WEB_TIER0_INT_HAMSTER', ): None,
        ('VLA_WEB_TIER0_INT_MULTI', ): None,
        ('VLA_WEB_TIER0_INVERTED_INDEX', ): ('vla-web-search.tier0-inverted-index', ),
        ('VLA_WEB_TIER0_KEYINV', ): ('vla-web-search.tier0-keyinv', ),
        (
            'VLA_YT_RTC_YT_ARNOLD_COLOCATION_1',
            'VLA_YT_RTC_YT_ARNOLD_COLOCATION_2',
            'VLA_YT_RTC_YT_ARNOLD_COLOCATION_3',
            'VLA_YT_RTC_YT_ARNOLD_COLOCATION_4'
        ): ('vla-yt-arnold-co-nodes-over-yp', 'vla-yt-arnold-co-solomon-bridge-over-yp'),
    },
}


class Instance(collections.namedtuple('Instance', ['group', 'node_id', 'port', 'cpu', 'mem', 'hdd', 'ssd'])):
    def __init__(self, group, node_id, port, cpu, mem, hdd, ssd):
        super(Instance, self).__init__(group, node_id, port, cpu, mem, hdd, ssd)
        self._pod_ref = None

    @property
    def iid(self):
        return '{}:{}:{}'.format(self.group, self.node_id, self.port)

    @property
    def pod_ref(self):
        raise NotImplementedError()
        return self._pod_ref

    def vcpu(self, factor):
        return self.cpu * 1000 * factor

    def set_pod_ref(self, pod_ref):
        raise NotImplementedError()
        self._pod_ref = pod_ref

    def __repr__(self):
        return (
            'Instance(group={}, node_id={}, port={}, '
            'cpu={}, mem={}, hdd={}, ssd={})'
        ).format(self.group, self.node_id, self.port,
                 self.cpu, int(self.mem), int(self.hdd), int(self.ssd))


class Pod(collections.namedtuple('Pod', ['pod_id', 'node_id'])):
    def __init__(self, pod_id, node_id):
        super(Pod, self).__init__(pod_id, node_id)
        self._podset_id = None
        self._cpu = 0
        self._mem = 0
        self._hdd = 0
        self._ssd = 0
        self._hdd_io = 0
        self._ssd_io = 0
        self._instance_ref = None

    @property
    def podset_id(self):
        return self._podset_id

    def set_podset_id(self, podset_id):
        self._podset_id = podset_id

    @property
    def cpu(self):
        return self._cpu

    def set_cpu(self, cpu):
        self._cpu = cpu

    @property
    def mem(self):
        return self._mem

    def set_mem(self, mem):
        self._mem = mem

    @property
    def hdd(self):
        return self._hdd

    def add_disk(self, kind, value):
        if kind == 'hdd':
            self._hdd += value
        elif kind == 'ssd':
            self._ssd += value

    @property
    def ssd(self):
        return self._ssd

    @property
    def hdd_io(self):
        return self._hdd_io

    def add_disk_io(self, kind, value):
        if kind == 'hdd':
            self._hdd_io += value
        elif kind == 'ssd':
            self._ssd_io += value

    def add_hdd_io(self, hdd_io):
        self._hdd_io += hdd_io

    @property
    def ssd_io(self):
        return self._ssd_io

    def add_ssd_io(self, ssd_io):
        self._ssd_io += ssd_io

    @property
    def instance_ref(self):
        raise NotImplementedError()
        return self._instance_ref

    def set_instance_ref(self, instance):
        raise NotImplementedError()
        self._instance_ref = instance

    def __repr__(self):
        return (
            'Pod(pod_id={}, node_id={}, podset_id={}, '
            'cpu={}, mem={}, hdd={}, ssd={})'
        ).format(self.pod_id, self.node_id, self._podset_id,
                 self.cpu, self.mem, self.hdd, self.ssd)


def main():
    args = parse_args()
    configure_logging()

    hosts_data = {}

    groups = {
        group_name
        for group_set in GROUPS_PODSETS[args.location].keys()
        for group_name in group_set
    }

    for group in groups:
        logging.info('Process %s group', group)
        for instance in gencfg_api.searcher_lookup_instances(group, args.topology):
            gencfg_instance = get_gencfg_instance(instance, group)
            if gencfg_instance.node_id not in hosts_data:
                hosts_data[gencfg_instance.node_id] = make_empty_host(
                    gencfg_instance.node_id
                )
            hosts_data[gencfg_instance.node_id]['instances'].append(gencfg_instance)

    with yp_utils.client(args.location) as client:
        pods_podsets = dict(
            client.select_objects(
                'pod',
                filter='[/meta/type]="pod"',
                selectors=['/meta/id', '/meta/pod_set_id']
            )
        )

        all_hosts = hosts_data.keys()
        parts = 10

        for part in range(parts):
            resources = client.select_objects(
                'resource',
                filter='[/meta/kind] in ("cpu", "memory", "disk") and [/meta/type]="resource" and [/meta/node_id] in ("{}")'.format('", "'.join(all_hosts[part::parts])),
                selectors=['/meta', '/spec', '/status']
            )
            for meta, spec, status in resources:
                if meta['kind'] == 'cpu':
                    process_cpu_resource(meta, spec, status, hosts_data[meta['node_id']], pods_podsets)
                elif meta['kind'] == 'memory':
                    process_mem_resource(meta, spec, status, hosts_data[meta['node_id']], pods_podsets)
                elif meta['kind'] == 'disk':
                    process_disk_resource(meta, spec, status, hosts_data[meta['node_id']], pods_podsets)
                else:
                    logging.error('Unknown resource:\nMETA: %s\nSPEC: %s\nSTATUS: %s', meta, spec, status)

    for host in hosts_data:
        test_import_exported(hosts_data[host])
        test_unknown_pods(hosts_data[host], GROUPS_PODSETS[args.location])
        # test_enought_resources_for_unlanded(hosts_data[host])
        test_wrong_guarantees(hosts_data[host], GROUPS_PODSETS[args.location])

    if args.dump_host and args.dump_host in hosts_data:
        pprint.pprint(hosts_data[args.dump_host])
    elif args.dump_host:
        logging.error('Host %s is unknown', args.dump_host)


def test_import_exported(host):
    exported_vcpu = sum(i.vcpu(host['cpu_factor']) for i in host['instances'])
    if 1 > (int(exported_vcpu) - host['total_capacity']['cpu']) < -1:
        logging.warn(
            'CPU export is not equal %s != %s (host %s)',
            int(exported_vcpu), host['total_capacity']['cpu'], host['fqdn']
        )

    exported_mem = sum(i.mem for i in host['instances'])
    if 1 > (host['total_capacity']['mem'] - exported_mem) < -1:
        logging.warn(
            'MEM export is not equal %s != %s (host %s)',
            int(exported_mem), host['total_capacity']['mem'], host['fqdn']
        )

    exported_hdd = sum(i.hdd for i in host['instances'])
    if 1 > (host['total_capacity']['hdd'] - exported_hdd) < -1:
        logging.warn(
            'HDD export is not equal %s != %s (host %s)',
            int(exported_hdd), host['total_capacity']['hdd'], host['fqdn']
        )

    exported_ssd = sum(i.ssd for i in host['instances'])
    if 1 > (host['total_capacity']['ssd'] - exported_ssd) < -1:
        logging.warn(
            'SSD export is not equal %s != %s (host %s)',
            int(exported_ssd), host['total_capacity']['ssd'], host['fqdn']
        )


def test_unknown_pods(host, groups_podsets):
    podset_groups = collections.defaultdict(set)

    for instance in host['instances']:
        for group_set, podsets in groups_podsets.iteritems():
            if instance.group in group_set and podsets:
                for podset in podsets:
                    podset_groups[podset].add(instance.group)

    unknown_pods = [
        pod
        for pod in host['pods'].itervalues()
        if not podset_groups[pod.podset_id]
    ]

    if unknown_pods:
        logging.warn('Unknown pods on host %s: %s', host['fqdn'], unknown_pods)


# TODO: rewrite, broken
def test_enought_resources_for_unlanded(host):
    unlanded = [i for i in host['instances'] if not i.pod_ref]
    required_vcpu = sum(i.vcpu(host['cpu_factor']) for i in unlanded)
    if required_vcpu > host['free']['cpu']:
        logging.warn(
            'Not enought cpu (required %s, free %s) at host %s for unlanded instances:\n%s',
            required_vcpu, host['free']['cpu'], host['fqdn'], pprint.pformat(unlanded)
        )
    required_mem = sum(i.mem for i in unlanded)
    if required_mem > host['free']['mem']:
        logging.warn(
            'Not enought memory (required %s, free %s) at host %s for unlanded instances:\n%s',
            required_mem, host['free']['mem'], host['fqdn'], pprint.pformat(unlanded)
        )
    required_hdd = sum(i.hdd for i in unlanded)
    if required_hdd > host['free']['hdd']:
        logging.warn(
            'Not enought hdd (required %s, free %s) at host %s for unlanded instances:\n%s',
            required_hdd, host['free']['hdd'], host['fqdn'], pprint.pformat(unlanded)
        )
    required_ssd = sum(i.ssd for i in unlanded)
    if required_ssd > host['free']['ssd']:
        logging.warn(
            'Not enought ssd (required %s, free %s) at host %s for unlanded instances:\n%s',
            required_ssd, host['free']['ssd'], host['fqdn'], pprint.pformat(unlanded)
        )


def test_wrong_guarantees(host, groups_podsets):
    for groups_set, podsets in groups_podsets.iteritems():
        if not podsets:
            continue

        exported_resources = dict.fromkeys(['cpu', 'mem', 'hdd', 'ssd'], 0)
        insts = set()
        for instance in host['instances']:
            if instance.group in groups_set:
                exported_resources['cpu'] += instance.vcpu(host['cpu_factor'])
                exported_resources['mem'] += instance.mem
                exported_resources['hdd'] += instance.hdd
                exported_resources['ssd'] += instance.ssd
                insts.add(instance)

        # exported_resources['hdd-io'] = 150 * 1024 ** 2 # TODO
        # exported_resources['ssd-io'] = 250 * 1024 ** 2 # TODO

        used_resources = dict.fromkeys(['cpu', 'mem', 'hdd', 'ssd', 'hdd-io', 'ssd-io'], 0)
        pods = set()
        for pod in host['pods'].itervalues():
            if pod.podset_id in podsets:
                used_resources['cpu'] += pod.cpu
                used_resources['mem'] += pod.mem
                used_resources['hdd'] += pod.hdd
                used_resources['ssd'] += pod.ssd
                used_resources['hdd-io'] += pod.hdd_io
                used_resources['ssd-io'] += pod.ssd_io
                pods.add(pod)

        changes = {}
        if exported_resources['cpu'] < used_resources['cpu']:
            changes['cpu'] = '{} -> {}'.format(used_resources['cpu'], int(exported_resources['cpu']))
        if exported_resources['mem'] < used_resources['mem']:
            changes['mem'] = '{} -> {}'.format(used_resources['mem'], int(exported_resources['mem']))
        if exported_resources['hdd'] < used_resources['hdd']:
            changes['hdd'] = '{} -> {}'.format(used_resources['hdd'], int(exported_resources['hdd']))
        if exported_resources['ssd'] < used_resources['ssd']:
            changes['ssd'] = '{} -> {}'.format(used_resources['ssd'], int(exported_resources['ssd']))

        if changes:
            logging.warn('%s: %s\n%s', list(pods), changes, list(insts))


def process_cpu_resource(meta, spec, status, host, pods_podsets):
    host['total_capacity']['cpu'] = spec['cpu']['total_capacity']
    host['cpu_factor'] = spec['cpu']['cpu_to_vcpu_factor']
    host['free']['cpu'] = status['free']['cpu']['guaranteed_capacity']
    for pod in status['scheduled_allocations']:
        if pod['pod_id'] not in host['pods']:
            host['pods'][pod['pod_id']] = Pod(pod['pod_id'], meta['node_id'])
        host['pods'][pod['pod_id']].set_cpu(pod['cpu']['capacity'])
        host['pods'][pod['pod_id']].set_podset_id(pods_podsets.get(pod['pod_id']))


def process_mem_resource(meta, spec, status, host, pods_podsets):
    host['total_capacity']['mem'] = spec['memory']['total_capacity']
    host['free']['mem'] = status['free']['memory']['guaranteed_capacity']
    for pod in status['scheduled_allocations']:
        if pod['pod_id'] not in host['pods']:
            host['pods'][pod['pod_id']] = Pod(pod['pod_id'], meta['node_id'])
        host['pods'][pod['pod_id']].set_mem(pod['memory']['capacity'])


def process_disk_resource(meta, spec, status, host, pods_podsets):
    if spec['disk']['storage_class'] == 'hdd':
        device_name, io_name = 'hdd', 'hdd-io',
    elif spec['disk']['storage_class'] == 'ssd':
        device_name, io_name = 'ssd', 'ssd-io'
    else:
        logging.error('Unknown disk resource: %s', meta['id'])
        return

    host['total_capacity'][device_name] = spec['disk']['total_capacity']
    host['total_capacity'][io_name] = spec['disk']['total_bandwidth']
    host['free'][device_name] = status['free']['disk']['capacity']
    host['free'][io_name] = status['free']['disk']['bandwidth']
    for pod in status['scheduled_allocations']:
        if pod['pod_id'] not in host['pods']:
            host['pods'][pod['pod_id']] = Pod(pod['pod_id'], meta['node_id'])
        host['pods'][pod['pod_id']].add_disk(device_name, pod['disk']['capacity'])
        host['pods'][pod['pod_id']].add_disk_io(device_name, pod['disk']['bandwidth'])


def find_pod(groups_podset, pods, instance):
    if not groups_podset[instance.group]:
        return None
    for pod in pods:
        if pod.instance_ref:
            continue
        if groups_podset[instance.group] == pod.podset_id:
            return pod


def make_empty_host(hostname):
    return {
        'fqdn': hostname,
        'cpu_factor': None,
        'instances': [],
        'pods': {},
        'total_capacity': {'cpu': 0, 'mem': 0,
                           'hdd': 0, 'ssd': 0,
                           'hdd-io': 0, 'ssd-io': 0},
        'free': {'cpu': 0, 'mem': 0,
                 'hdd': 0, 'ssd': 0,
                 'hdd-io': 0, 'ssd-io': 0},

    }


def get_gencfg_instance(instance, group):
    return Instance(
        group=group,
        node_id=instance['hostname'],
        port=instance['port'],
        cpu=instance['porto_limits']['cpu_cores_guarantee'],
        mem=instance['porto_limits']['memory_guarantee'],
        hdd=get_gencfg_disksize(instance['storages'], 'hdd') * 1024 ** 3,
        ssd=get_gencfg_disksize(instance['storages'], 'ssd') * 1024 ** 3,
    )


def get_gencfg_disksize(storage, disk_type):
    sizes = [
        disk['size']
        for disk in storage.values()
        if disk['partition'] == disk_type
    ]

    assert len(sizes) < 2

    return sizes[-1] if sizes else .0


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--topology', required=True)
    parser.add_argument('--location', required=True)
    parser.add_argument('--dump-host', required=False)

    return parser.parse_args()


def configure_logging():
    logging.getLogger().setLevel(logging.INFO)


if __name__ == '__main__':
    main()
