#include <infra/callisto/protos/deploy/tables.pb.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/wait_for_tablets_state.h>
#include <mapreduce/yt/io/proto_table_writer.h>

#include <library/cpp/getoptpb/getoptpb.h>

#include "infra/callisto/tools/create_deploy_table/options.pb.h"

template <class T>
TVector<TString> GetKeyFields() {
    const T msg;
    TVector<TString> fields;

    const auto msg_descr = msg.GetDescriptor();
    for (auto i = 0; i < msg_descr->field_count(); ++i) {
        const auto field_opts = msg_descr->field(i)->options();

        if (field_opts.HasExtension(NYT::key_column_name)) {
            fields.push_back(field_opts.GetExtension(NYT::key_column_name));
        }
    }

    return fields;
}

template <class T>
void CreateTable(NYT::IClientPtr client, const TString& table_path) {
    if (not client->Exists(table_path)) {
        Cerr << "Create table " << table_path << Endl;

        const auto schema = NYT::CreateTableSchema<T>(GetKeyFields<T>());

        client->Create(
            table_path, NYT::NT_TABLE,
            NYT::TCreateOptions()
                .Force(true)
                .Attributes(
                    NYT::TNode()("dynamic", true)("schema", schema.ToNode())));

        client->MountTable(table_path);
        WaitForTabletsState(client, table_path, NYT::TS_MOUNTED);
    } else {
        Cerr << "Table " << table_path << " exists" << Endl;
    }
}

int main(int argc, const char **argv) {
    TArgs args = NGetoptPb::GetoptPbOrAbort(argc, argv, {.UseLetterHelpShortcut = true});
    auto yt_client = NYT::CreateClient(args.GetYtCluster());

    switch (args.GetTableType()) {
        case TArgs::TTARGET:
            CreateTable<NPlutonium::TPodTarget>(yt_client, args.GetTablePath());
            break;
        case TArgs::TSTATUS:
            CreateTable<NPlutonium::TPodStatus>(yt_client, args.GetTablePath());
            break;
        default:
            Cerr << "Unknown table type" << Endl;
            break;
    }

    return 0;
}
