#include "enums.h"

#include <infra/callisto/tools/deploy_viewer/protos/config.pb.h>
#include <infra/callisto/tools/deploy_viewer/protos/deployment.pb.h>
#include <infra/callisto/tools/deploy_viewer/protos/mapping_entry.pb.h>
#include <infra/callisto/tools/deploy_viewer/protos/tracker_entry.pb.h>

#include <infra/callisto/reports/proto/report.pb.h>

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/msgpack2json/msgpack2json.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/protobuf/util/pb_io.h>

#include <kernel/doom/erasure/codec.h>

#include <search/base/blob_storage/codec.h>

#include <mapreduce/yt/interface/client.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/generic/hash_set.h>
#include <util/generic/size_literals.h>
#include <util/generic/xrange.h>
#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/string/subst.h>
#include <util/system/fs.h>

#include <google/protobuf/util/json_util.h>
#include <google/protobuf/message.h>

#include <contrib/libs/re2/re2/re2.h>

#include <fmt/format.h>


using infra::callisto::TReportContainerMessage;

namespace NInfra::NDeployViewer {

template <typename T>
requires (std::is_floating_point_v<T>)
auto PrettyFloat(T value) {
    return Prec(value, PREC_POINT_DIGITS, 2);
}

auto Percents(double value) {
    return PrettyFloat(100.0 * value);
}

template <typename T>
auto Percents(T a, T b) {
    return Percents(static_cast<double>(a) / b);
}

template <typename T>
auto Ratio(T a, T b) {
    return PrettyFloat(static_cast<double>(a) / b);
}

////////////////////////////////////////////////////////////////////////////////

TString GlobalTimestamp = "1577050752";
bool GlobalCacheEnabled = true;

void CheckResponseErrors(const NNeh::TResponseRef& response, const TStringBuf url) {
    if (!response) {
        throw yexception() << "Unknown error when fetching url '" << url << "'" << Endl;
    } else if (response->IsError()) {
        throw yexception() << "Error when fetching url '" << url << "': " << response->GetErrorText() << Endl;
    }
}

TReportContainerMessage FetchReports(const TStringBuf url) {
    NNeh::TMessage message;
    message.Addr = url;
    NNeh::NHttp::MakeFullRequest(message, "Accept: application/protobuf\n", "", "application/x-www-form-urlencoded", NNeh::NHttp::ERequestType::Get);

    // arc+extinfo answer is 800 MB
    NNeh::TResponseRef resp = NNeh::Request(message)->Wait(TDuration::Seconds(180));
    CheckResponseErrors(resp, url);

    TReportContainerMessage reports;
    Y_ENSURE(reports.ParseFromString(resp->Data));

    return reports;
}

TVector<TStringBuf> GetDeployerTagByLocation(const TStringBuf location) {
    if (location == "sas") {
        return { "SAS_WEB_DEPLOY", "sas-web-search" };
    }

    if (location == "man") {
        return { "MAN_WEB_DEPLOY" };
    }

    if (location == "vla") {
        return { "VLA_WEB_DEPLOY" };
    }

    if (location == "pip") {
        return { "VLA_WEB_BASE_PIP_DEPLOY" };
    }

    if (location == "builders_tier1") {
        return { "VLA_WEB_TIER1_BUILD" };
    }

    if (location == "erasure_beta") {
        return { "VLA_ERASURE_BETA_DEPLOY", "SAS_SSMIKE_DEV", "DEV_DEPLOY" };
    }

    throw yexception() << "Unknown location: " << location;
}

TReportContainerMessage FetchDeployerReportsByLocation(const TStringBuf location) {
    TVector<TStringBuf> tags = GetDeployerTagByLocation(location);
    TReportContainerMessage result;
    for (TStringBuf tag : tags) {
        TStringBuilder url;
        url << "https://";
        if (location == "pip" || location == "builders_tier1") {
            url << "vla";
        } else if (location == "erasure_beta") {
            url << "sas";
        } else {
            url << location;
        }
        url << "-reports-v2-cajuper.n.yandex-team.ru/with_tags?tag=" << tag << "&min_uptime=300";
        TReportContainerMessage reports = FetchReports(url);
        result.MergeFrom(reports);
    }
    return result;
}

TString GetCacheFileName(const TStringBuf location) {
    return TStringBuilder() << "cache/" << location << "." << GlobalTimestamp;
}

TReportContainerMessage FetchCached(const TStringBuf location) {
    TString cacheFile = GetCacheFileName(location);
    if (GlobalCacheEnabled && NFs::Exists(cacheFile)) {
        TFileInput input(cacheFile);

        TReportContainerMessage message;
        Y_ENSURE(message.ParseFromString(input.ReadAll()));
        return message;
    } else {
        TReportContainerMessage message = FetchDeployerReportsByLocation(location);

        NFs::MakeDirectoryRecursive("cache");
        TFileOutput output(cacheFile);
        output.Write(message.SerializeAsString());
        return message;
    }
}

TString JsonToString(const NJson::TJsonValue& json) {
    TStringStream s;
    NJson::TJsonWriterConfig config;
    config.SortKeys = true;
    config.FormatOutput = true;
    NJson::TJsonWriter{&s, config}.Write(&json);
    return s.Str();
}

TString ProtoToJson(const google::protobuf::Message& message) {
    TString buffer;
    auto status = google::protobuf::util::MessageToJsonString(message, &buffer);
    Y_ENSURE(status.ok(), "Failed to convert protobuf to json: " << status.ToString());
    return buffer;
}

class TResourceName {
public:
    TString Shard;
    TString Tier;
    ui32 Chunk = Max<ui32>();
    ui32 Part = Max<ui32>();
    TString Timestamp;
    TString Name;
    TString ReplicationType;
    TMaybe<ERemoteStoragePartSubResource> SubresourceKind;

    static TResourceName Parse(const TStringBuf resourceName) {
        TVector<TStringBuf> parts = StringSplitter(resourceName).Split('/').ToList<TStringBuf>();

        TVector<TStringBuf> nameParts = StringSplitter(parts[0]).Split('-').ToList<TStringBuf>();

        TResourceName parsedName;
        parsedName.Name = resourceName;
        parsedName.Shard = parts[0];
        parsedName.Timestamp = nameParts[nameParts.size() - 1];

        if (nameParts[0] == "primus") {
            parsedName.Tier = nameParts[1];

            if (parts.size() == 4) {
                parsedName.ReplicationType = parts[1];
                parsedName.Chunk = FromString(parts[2]);
                parsedName.Part = FromString(parts[3]);
            } else if (parts.size() == 5) {
                parsedName.ReplicationType = parts[1];
                parsedName.Chunk = FromString(parts[2]);
                parsedName.Part = FromString(parts[3]);
                parsedName.SubresourceKind = FromString(parts[4]);
            } else {
                Y_ENSURE(parts.size() <= 2, "Unsupported resource name: " << resourceName);
            }
        } else if (nameParts[0] == "rearr") {
            parsedName.Tier = "MsUserDataTier";
        } else {
            parsedName.Tier = nameParts[0];
        }

        return parsedName;
    }
};

class TResource {
public:
    TString Namespace;
    TResourceName Name;
    EResourceStatus Status = EResourceStatus::IDLE;
};

class TAgent {
public:
    TString Host;
    ui32 Port = 0;
};

class TReport {
public:
    TAgent Agent;
    TVector<TResource> Resources;
    ui64 FreeSpace = 0;
    ui64 Timestamp = 0;
};

class TReportsSet {
public:
    TVector<TReport> Reports;
};

TReportsSet ParseReports(TReportContainerMessage reportsMessage, ui64 minTimestamp) {
    TReportsSet reportsSet;

    size_t numReportsSkipped = 0;

    for (const infra::callisto::TReportMessage& reportMessage : reportsMessage.Getreports()) {
        const TString& reportsData = reportMessage.data();

        msgpack::unpacked msg;
        msgpack::unpack(msg, reportsData.data(), reportsData.length());

        msgpack::object obj = msg.get();
        NJson::TJsonValue jsonValue;
        NMsgpack2Json::Msgpack2Json(obj, &jsonValue);

        TReport report;

        report.Timestamp = reportMessage.timestamp();

        TAgent agent;

        agent.Host = reportMessage.Gethost();
        agent.Port = reportMessage.Getport();

        report.Agent = std::move(agent);

        if (report.Timestamp < minTimestamp) {
            ++numReportsSkipped;
            //Cerr << "Report from " << report.Agent.Host << " skipped due to timestamp filter: " << report.Timestamp << " vs " << minTimestamp << Endl;
            continue;
        }

        for (const NJson::TJsonValue& it : jsonValue["resources"].GetArray()) {
            TResource resource;
            resource.Namespace = it["namespace"].GetString();
            resource.Name = TResourceName::Parse(it["name"].GetString());
            resource.Status = FromString<EResourceStatus>(it["status"].GetString());
            report.Resources.push_back(std::move(resource));
        }

        report.FreeSpace = jsonValue["freespace"].GetInteger();

        reportsSet.Reports.push_back(std::move(report));
    }

    if (numReportsSkipped > 0) {
        Cerr << "NOTE: Skipped " << numReportsSkipped << " / " << reportsMessage.reportsSize() << " reports due to timestamp filter" << Endl;
    }

    return reportsSet;
}

void DumpReportsSet(const TReportsSet& reportsSet, const TString& timestamp, const TString& /*location*/, ui64 /*topSize*/) {
    for (const TReport& report : reportsSet.Reports) {
        for (const TResource& resource : report.Resources) {
            if (resource.Name.Timestamp != timestamp) {
                continue;
            }
            Cout << "agent=" << report.Agent.Host << ":" << report.Agent.Port
                << "\tfreespace=" << report.FreeSpace
                << "\ttimestamp=" << report.Timestamp
                << "\tnamespace=" << resource.Namespace
                << "\tstatus=" << ToString(resource.Status)
                << "\tshard=" << resource.Name.Shard
                << "\ttier=" << resource.Name.Tier
                << "\tchunk=" << resource.Name.Chunk
                << "\tpart=" << resource.Name.Part
                << "\ttimestamp=" << resource.Name.Timestamp
                << "\tname=" << resource.Name.Name
                << "\titype=" << resource.Name.ReplicationType
                << "\tsubresource=" << resource.Name.SubresourceKind.GetOrElse(ERemoteStoragePartSubResource::None)
                << '\n';
        }
    }
}

struct TSimpleChunkStats {
    ui64 CountTotal() const {
        return CountIdle_ + CountDownloading_ + CountPrepared_;
    }

    ui64 CountPrepared() const {
        return CountPrepared_;
    }

    ui64 CountUnprepared() const {
        return CountTotal() - CountPrepared();
    }

    void Add(EResourceStatus status) {
        switch (status) {
            case EResourceStatus::IDLE:
                ++CountIdle_;
                break;
            case EResourceStatus::DOWNLOADING:
                ++CountDownloading_;
                break;
            case EResourceStatus::PREPARED:
                ++CountPrepared_;
                break;
        }
    }

    void Print(IOutputStream& out, TStringBuf tier, TStringBuf chunk, ui64 maxTotal) const {
        out << "tier=" << tier << '\t'
            << "chunk=" << chunk << '\t'
            << "max_total=" << maxTotal << '\t'
            << "total_reported=" << CountTotal() << '\t'
            << "prepared=" << CountPrepared_ << '\t'
            << "downloading=" << CountDownloading_ << '\t'
            << "idle=" << CountIdle_ << '\n';
    }

    void PrintHost(IOutputStream& out, TStringBuf host) const {
        out << "host=" << host << '\t'
            << "total_reported=" << CountTotal() << '\t'
            << "prepared=" << CountPrepared_ << '\t'
            << "downloading=" << CountDownloading_ << '\t'
            << "idle=" << CountIdle_ << '\n';
    }

private:
    ui64 CountIdle_ = 0;
    ui64 CountDownloading_ = 0;
    ui64 CountPrepared_ = 0;
};

struct TSimpleTierStats {
    void Add(const TString& shard, ui64 chunkId, EResourceStatus status) {
        StatsByChunks_[std::make_pair(shard, chunkId)].Add(status);
    }

    void ShowTopUnpreparedShards(size_t topSize, IOutputStream& out, TStringBuf tier) const {
        using TStatPair = std::pair<std::pair<TString, ui64>, TSimpleChunkStats>;
        auto lessPrepared = [](const TStatPair& a, const TStatPair& b) {
            return a.second.CountPrepared() < b.second.CountPrepared() ||
                a.second.CountPrepared() == b.second.CountPrepared() && a.second.CountTotal() < b.second.CountTotal();
        };
        ui64 maxTotal = 0;
        TVector<TStatPair> sortedStats;
        for (const auto& statPair : StatsByChunks_) {
            maxTotal = std::max(maxTotal, statPair.second.CountTotal());
            sortedStats.push_back(statPair);
        }
        std::sort(sortedStats.begin(), sortedStats.end(), lessPrepared);
        out << "--- Tier " << tier << " ---\n";
        for (size_t i = 0; i < topSize && i < sortedStats.size(); ++i) {
            const TStatPair& stat = sortedStats[i];
            const TString chunkName = stat.first.second == Max<ui32>() ? stat.first.first : TString::Join(stat.first.first, '/', ToString(stat.first.second));
            stat.second.Print(out, tier, chunkName, maxTotal);
        }
    }

private:
    THashMap<std::pair<TString, ui64>, TSimpleChunkStats> StatsByChunks_;
};

void ShowTopUnpreparedShards(const TReportsSet& reportsSet, const TString& timestamp, const TString& location, ui64 topSize) {
    THashMap<std::pair<TString, TString>, TSimpleTierStats> statsByTier;
    for (const TReport& report : reportsSet.Reports) {
        for (const TResource& resource : report.Resources) {
            if (resource.Name.Timestamp != timestamp) {
                continue;
            }
            statsByTier[std::make_pair(resource.Name.Tier, resource.Name.ReplicationType)].Add(resource.Name.Shard, resource.Name.Chunk, resource.Status);
        }
    }

    Cout << "=== Location " << location << " ===\n";

    for (auto [tier, tierStats] : statsByTier) {
        const TString tierName = tier.second.empty() ? tier.first : TString::Join(tier.first, '/', tier.second);
        tierStats.ShowTopUnpreparedShards(topSize, Cout, tierName);
    }
}

void ShowTopUnpreparedHosts(const TReportsSet& reportsSet, const TString& timestamp, const TString& location, ui64 topSize) {
    THashMap<std::pair<TString, ui32>, TSimpleChunkStats> statsByHost;
    for (const TReport& report : reportsSet.Reports) {
        for (const TResource& resource : report.Resources) {
            if (resource.Name.Timestamp != timestamp) {
                continue;
            }
            statsByHost[std::make_pair(report.Agent.Host, report.Agent.Port)].Add(resource.Status);
        }
    }

    Cout << "=== Location " << location << " ===\n";

    using TStatPair = std::pair<std::pair<TString, ui32>, TSimpleChunkStats>;
    auto lessPrepared = [](const TStatPair& a, const TStatPair& b) {
        return a.second.CountUnprepared() > b.second.CountUnprepared() ||
            a.second.CountUnprepared() == b.second.CountUnprepared() && a.second.CountTotal() < b.second.CountTotal();
    };
    TVector<TStatPair> sortedStats(statsByHost.begin(), statsByHost.end());
    std::sort(sortedStats.begin(), sortedStats.end(), lessPrepared);
    for (size_t i = 0; i < topSize && i < sortedStats.size(); ++i) {
        const TStatPair& stat = sortedStats[i];
        const TString host = TString::Join(stat.first.first, ':', ToString(stat.first.second));
        stat.second.PrintHost(Cout, host);
    }
}

class TChunkStats {
public:
    TVector<ERemoteStoragePartSubResources> PartReplicasPrepared;
    ui32 NumPartsPrepared;

    void AddPart(ui32 part, TMaybe<ERemoteStoragePartSubResource> subResource = {}) {
        if (PartReplicasPrepared.size() <= part) {
            PartReplicasPrepared.resize(part + 1);
        }
        auto subResourceMask = subResource.GetOrElse(ERemoteStoragePartSubResource::Ready);
        auto prevMask = PartReplicasPrepared[part];
        auto currentMask = prevMask | subResourceMask;
        PartReplicasPrepared[part] |= currentMask;
        if (prevMask != ERemoteStoragePartSubResource::Ready && currentMask == ERemoteStoragePartSubResource::Ready) {
            ++NumPartsPrepared;
        }
    }

    bool HasPart(ui32 part) const {
        if (part >= PartReplicasPrepared.size()) {
            return false;
        }
        return PartReplicasPrepared[part] & ERemoteStoragePartSubResource::Ready;
    }
};

class TShardStats {
public:
    ui32 LocalReplicsPrepared = 0;
    TVector<TChunkStats> ChunkStats;
};

class TTierStats {
public:
    TString Name;
    TString Prefix;
    TString Format;
    ui32 NumShardsInGroup = 0;
    ui32 NumGroups = 0;
    ui32 NumReplics = 0;
    ui32 NumChunks = 0;
    ui32 NumParts = 0;
    ui32 ChunkModulo = Max<ui32>();
    const NDoom::ICodec* Codec = nullptr;

    THashMap<TString, TShardStats> StatsByShard;

public:
    ui64 TotalNumParts() const {
        return ui64{NumShardsInGroup} * NumGroups * NumReplics * NumChunks * NumParts;
    }

public:
    static TTierStats FromProto(const TTierConfig& config, TTierConfig::ELocationType ctype, const TString& timestamp) {
        auto* location = FindIfPtr(config.GetReplication(), [ctype](const auto& location) {
            return location.GetLocation() == ctype || location.GetLocation() == TTierConfig::All;
        });
        ui32 numReplics = location ? location->GetNumReplics() : 1;

        const NDoom::ICodec* codec = &NBlobStorage::GetCodec(config.GetCodec());

        TTierStats stats{
            .Name = config.GetName(),
            .Prefix = config.GetPrefix(),
            .Format = config.GetFormat(),
            .NumShardsInGroup = config.GetNumShardsInGroup(),
            .NumGroups = config.GetNumGroups(),
            .NumReplics = numReplics,
            .NumChunks = config.GetNumChunks(),
            .NumParts = static_cast<ui32>(codec->GetTotalPartCount()),
            .ChunkModulo = config.HasChunkModulo() ? config.GetChunkModulo() : Max<ui32>(),
            .Codec = codec,
        };
        stats.FillEmptyShardStats(timestamp);

        return stats;
    }

    void FillEmptyShardStats(TString timestamp) {
        TVector<TString> shardNames;

        for (ui32 group = 0; group < NumGroups; ++group) {
            for (ui32 shardIndex = 0; shardIndex < NumShardsInGroup; ++shardIndex) {
                std::string shardName = fmt::format(Format.ConstRef()
                    , fmt::arg("prefix", Prefix.ConstRef())
                    , fmt::arg("name", Name.ConstRef())
                    , fmt::arg("group", group)
                    , fmt::arg("index", shardIndex)
                    , fmt::arg("timestamp", timestamp.ConstRef())
                );
                shardNames.emplace_back(std::move(shardName));
            }
        }

        for (const TString& shardName : shardNames) {
            StatsByShard[shardName] = TShardStats();
            if (NumChunks > 0) {
                StatsByShard[shardName].ChunkStats.resize(NumChunks);
            }
        }
    }
};

TVector<TTrackerEntry> FetchTrackerEntries() {
    TString cacheFile = GetCacheFileName("tracker");
    if (GlobalCacheEnabled && NFs::Exists(cacheFile)) {
        TFileInput input(cacheFile);

        TVector<TTrackerEntry> entries;
        ::Load(&input, entries);
        return entries;
    } else {
        TVector<TTrackerEntry> entries;

        NYT::IClientPtr client = NYT::CreateClient("arnold");
        for (TStringBuf suffix : {"/PlatinumTier0"sv, "/WebTier0"sv, "/AttributeWebTier0"sv, "/MsuseardataJupiterTier0"sv}) {
            TString trackerTablePath = "//home/cajuper/tracker/web/prod/" + GlobalTimestamp + suffix + "/table";
            if (client->Exists(trackerTablePath)) {
                NYT::TTableReaderPtr<TTrackerEntry> reader = client->CreateTableReader<TTrackerEntry>(trackerTablePath);

                for (; reader->IsValid(); reader->Next()) {
                    entries.push_back(reader->MoveRow());
                }
            } else {
                Cout << "WARNING: tracker table '" << trackerTablePath << " does not exist (maybe there were no shards yet)" << Endl;
            }
        }

        TFileOutput output(cacheFile);
        ::Save(&output, entries);
        return entries;
    }
}

class TLazyTracker {
public:
    TVector<TTrackerEntry> Entries;

    void MaybeFetch() {
        if (!Fetched_) {
            Fetched_ = true;
            Entries = FetchTrackerEntries();
            BuildIndices();
        }
    }

    TTrackerEntry* GetEntryByName(const TString& name) {
        MaybeFetch();
        ui32* index = EntryIndexByName_.FindPtr(name);
        return index ? &Entries[*index] : nullptr;
    }

private:
    bool Fetched_ = false;
    THashMap<TString, ui32> EntryIndexByName_;

    void BuildIndices() {
        for (ui32 index = 0; index < Entries.size(); ++index) {
            EntryIndexByName_[Entries[index].GetName()] = index;
        }
    }
};

TLazyTracker GlobalLazyTracker;

struct TShowStatsLimits {
    ui32 MaxShardsNotInTracker = 10;
    ui32 MaxShardsByLocalReplics = 10;
    ui32 MaxChunksByPartsCount = 10;
    ui32 MaxDeployersWithNotReadyResources = 10;
};

void FillStatsByTier(const TConfig& config, TMap<TString, TTierStats>& stats, TTierConfig::ELocationType ctype, const TString& timestamp) {
    for (auto&& tier : config.GetTiers().GetTier()) {
        stats[tier.GetName() + tier.GetSuffix()] = TTierStats::FromProto(tier, ctype, timestamp);
    }
}

void ShowErasureStats(const TTierStats& tierStats) {
    // Skip identity codec
    const NDoom::ICodec* codec = tierStats.Codec;
    if (!codec || codec->GetTotalPartCount() == 1) {
        return;
    }

    ui32 totalDataParts = 0;
    ui32 recoverableDataParts = 0;
    ui32 definitelyLostDataParts = 0;
    ui32 requestedParts = 0;
    TMap<ui32, ui32> requestsCount;

    NErasure::TPartIndexList erased;
    TVector<bool> loaded;

    for (auto&& [shard, shardStats] : tierStats.StatsByShard) {
        for (ui32 chunk : xrange(tierStats.NumChunks)) {
            const TChunkStats& chunkStats = shardStats.ChunkStats.at(chunk);

            totalDataParts += codec->GetDataPartCount();
            for (ui32 part : xrange(codec->GetDataPartCount())) {
                // Try to model basesearch behaviour
                erased.clear();
                loaded.assign(codec->GetTotalPartCount(), false);
                ui32 requests = 0;
                for (;;) {
                    auto partList = codec->GetRepairIndices(part, erased);
                    if (!partList) {
                        ++definitelyLostDataParts;
                        break;
                    }

                    bool hasAll = true;
                    for (int part : *partList) {
                        requests += !std::exchange(loaded[part], true);
                        bool hasPart = chunkStats.HasPart(part);
                        hasAll &= hasPart;
                        if (!hasPart) {
                            erased.push_back(part);
                        }
                    }

                    if (hasAll) {
                        ++recoverableDataParts;
                        break;
                    }
                }
                requestsCount[requests]++;
                requestedParts += requests;
            }
        }
    }
    Y_VERIFY(totalDataParts == recoverableDataParts + definitelyLostDataParts);

    Cout << NColorizer::CYAN << "  Erasure stats:\n" << NColorizer::RESET;

    Cout
        << "    Recoverable " << recoverableDataParts << " / " << totalDataParts
        << " (" << Percents(recoverableDataParts, totalDataParts) << "%) parts"
        << ", definitely lost " << definitelyLostDataParts
        << " (" << Percents(definitelyLostDataParts, totalDataParts) << "%) parts"
        << ", estimated multiplication: "
        << Ratio(requestedParts, totalDataParts) << 'x' << Endl;

    for (auto [numParts, count] : requestsCount) {
        Cout << "      " << count <<  " requests with " << numParts << " parts" << Endl;
    }
}

TDeploymentStats ShowStats(const TConfig& config, const TReportsSet& reportsSet, const TStringBuf location, const TShowStatsLimits& showStatsLimits) {
    TDeploymentStats deploymentStats;
    TMap<TString, TTierStats> statsByTier;

    FillStatsByTier(config, statsByTier, location == "pip" ? TTierConfig::Pip : TTierConfig::Prod, GlobalTimestamp);

    size_t numNotInTracker = 0;
    size_t numNotShown = 0;
    for (const auto& [shardName, shardStats] : statsByTier["WebTier0"].StatsByShard) {
        if (!GlobalLazyTracker.GetEntryByName(shardName + "/local")) {
            if (numNotInTracker <= showStatsLimits.MaxShardsNotInTracker) {
                Cout << "Tracker has no shard " << shardName << Endl;
            } else {
                ++numNotShown;
            }
            ++numNotInTracker;
        }
    }
    if (numNotShown > 0) {
        Cout << "(" << numNotShown << " more shards not in tracker are not shown)" << Endl;
    }
    Cout << "Total " << numNotInTracker << " shards are not in tracker" << Endl;

    TMaybe<re2::RE2> ignoredNamespaceRegexp;
    if (auto re = config.GetNamespaces().GetIgnoredRegexp()) {
        ignoredNamespaceRegexp.ConstructInPlace(re);
    }
    size_t ignoredNamespaceResources = 0;

    THashSet<TString> deployersWithNotReadyResourceFromState;
    THashMap<TString, ui32> unknownTiers;
    for (const TReport& report : reportsSet.Reports) {
        for (const TResource& resource : report.Resources) {
            //Cout << "Have shard '" << resource.Name.Shard << "' status '" << resource.Status << "' tier '" << resource.Name.Tier << " chunk " << resource.Name.Chunk << " timestamp " << resource.Name.Timestamp << Endl;

            if (resource.Name.Timestamp != GlobalTimestamp) {
                continue;
            }
            if (ignoredNamespaceRegexp && re2::RE2::PartialMatch(resource.Namespace, *ignoredNamespaceRegexp)) {
                ++ignoredNamespaceResources;
                continue;
            }

            if (resource.Status != EResourceStatus::PREPARED) {
                if (statsByTier.contains(resource.Name.Tier) && resource.Name.Chunk != Max<ui32>()) {
                    deployersWithNotReadyResourceFromState.insert(report.Agent.Host + ":" + ToString(report.Agent.Port));
                }

                continue;
            }

            if (!statsByTier.contains(resource.Name.Tier)) {
                unknownTiers[resource.Name.Tier]++;
                continue;
            }

            if (resource.Name.Chunk == Max<ui32>()) {
                auto& tierStats = statsByTier.at(resource.Name.Tier);
                if (tierStats.StatsByShard.contains(resource.Name.Shard)) {
                    ++tierStats.StatsByShard.at(resource.Name.Shard).LocalReplicsPrepared;
                } else {
                    Cerr << "Unknown tier+shard pair: " << resource.Name.Tier << "+" << resource.Name.Shard << "\n";
                }
            } else {
                const TString tier = resource.Name.Tier + "_" + resource.Name.ReplicationType;
                if (statsByTier.contains(tier)) {
                    auto& tierStats = statsByTier.at(tier);
                    if (tierStats.StatsByShard.contains(resource.Name.Shard)) {
                        auto& chunkStats = tierStats.StatsByShard.at(resource.Name.Shard).ChunkStats;
                        const ui32 chunk = resource.Name.Chunk % tierStats.ChunkModulo;
                        if (chunkStats.size() <= chunk) {
                            chunkStats.resize(chunk + 1);
                        }
                        chunkStats.at(chunk).AddPart(resource.Name.Part, resource.Name.SubresourceKind);
                    } else {
                        Cerr << "Unknown tier+shard pair: " << tier << "+" << resource.Name.Shard << "\n";
                    }
                } else {
                    Cerr << "Unknown tier: " << tier << "\n";
                }
            }
        }
    }

    for (auto&& [tier, numResources] : unknownTiers) {
        Cerr << "Skipped unknown tier " << tier << " with " << numResources << " resources\n";
    }
    if (ignoredNamespaceResources > 0) {
        Cerr << "Skipped " << ignoredNamespaceResources << " resources due to namespace filter\n";
    }

    for (const auto& [tier, tierStats] : statsByTier) {
        Cout << NColorizer::CYAN << "Stats for " << tier << NColorizer::RESET << ":" << Endl;

        TMap<ui32, TVector<TString>> ShardsByLocalReplicsCount;
        THashMap<ui32, TVector<TString>> ChunksByPartsCount;

        ui64 totalParts = tierStats.TotalNumParts();
        ui64 preparedParts = 0;

        for (const auto& [shard, shardStats] : tierStats.StatsByShard) {
            ShardsByLocalReplicsCount[shardStats.LocalReplicsPrepared].push_back(shard);

            for (ui32 chunk = 0; chunk < tierStats.NumChunks; ++chunk) {
                ChunksByPartsCount[shardStats.ChunkStats[chunk].NumPartsPrepared].push_back(TStringBuilder() << shard << "/remote_storage/" << chunk);
                preparedParts += shardStats.ChunkStats[chunk].NumPartsPrepared;
            }
        }

        Y_ENSURE(preparedParts <= totalParts);

        if (tierStats.NumChunks == 0) {
            ui64 totalReplics = tierStats.NumReplics * tierStats.StatsByShard.size();
            ui64 preparedReplics = 0;

            for (const auto& [numLocalReplics, shardNames] : ShardsByLocalReplicsCount) {
                if (shardNames.empty()) {
                    continue;
                }

                Cout << "  " << shardNames.size() << " shards are in " << numLocalReplics << " replics" << Endl;
                preparedReplics += shardNames.size() * numLocalReplics;

                if (shardNames.size() <= showStatsLimits.MaxShardsByLocalReplics) {
                    Cout << "    These shards are:" << "\n";
                    for (const auto& shardName : shardNames) {
                        Cout << "      " << shardName;

                        TTrackerEntry* entry = GlobalLazyTracker.GetEntryByName(shardName + ((tier == "WebTier0") ? "/local" : ""));
                        if (!entry) {
                            GlobalLazyTracker.GetEntryByName(shardName);
                        }
                        if (!entry) {
                            Cout << " [not present in tracker]";
                        } else {
                            Cout << " [in tracker since " << entry->GetTime() << ": " << entry->GetRbtorrent() << "]";
                        }

                        Cout << "\n";
                    }
                }
            }

            Cout
                << "  total " << preparedReplics << " replics prepared of " << totalReplics
                << " (" << Percents(preparedReplics, totalReplics) << "%)" << Endl;
        }

        for (ui32 numParts = 0; numParts <= tierStats.NumParts; ++numParts) {
            if (!ChunksByPartsCount.contains(numParts)) {
                continue;
            }

            TVector<TString> shardChunkNames = ChunksByPartsCount.at(numParts);

            TDistributionPoint* point = deploymentStats.AddWebTier1PartDistribution();
            point->SetValue(numParts);
            point->SetCount(shardChunkNames.size());

            Cout << "  " << point->GetCount() << " chunks have " << point->GetValue() << " parts prepared" << Endl;

            if (!shardChunkNames.empty() && shardChunkNames.size() <= showStatsLimits.MaxChunksByPartsCount) {
                Cout << "    These chunks are:" << Endl;
                for (auto&& name : shardChunkNames) {
                    Cout << "      " << name << Endl;
                }
            }
        }

        if (tierStats.NumChunks != 0) {
            Cout << "  total " << preparedParts << " parts prepared of " << totalParts << " (" << Percents(preparedParts, totalParts) << "%)" << Endl;
        }

        ShowErasureStats(tierStats);
    }

    Cout << NColorizer::CYAN << "Deployer stats:" << NColorizer::RESET << Endl;
    Cout << "  " << deployersWithNotReadyResourceFromState.size() << " deployers have 'idle' or 'downloading' resources" << Endl;
    if (deployersWithNotReadyResourceFromState.size() <= showStatsLimits.MaxDeployersWithNotReadyResources) {
        Cout << "    These deployers are:" << Endl;
        for (const TString& hostPort : deployersWithNotReadyResourceFromState) {
            Cout << "      " << hostPort << Endl;

            if (false) {
                TVector<TString> agentParts = StringSplitter(hostPort).Split('-').ToList<TString>();
                TString agentNoMtn = agentParts[0] + "-" + agentParts[1] + ".search.yandex.net";

                Cout << "Agent " << hostPort << " status (" << agentNoMtn << "):" << Endl;

                //system(("ssh " + agentNoMtn + " 'ps -aux | grep skybone-dl'").data());
                system(("ssh " + agentNoMtn + " df /ssd").data());
            }
        }
    }

    return deploymentStats;
}

void FindResources(const TReportsSet& reportsSet, const TVector<TString> names) {
    TVector<TString> agentsToPing;

    THashSet<TString> resourcesNotFound(names.begin(), names.end());

    for (const TReport& report : reportsSet.Reports) {
        for (const TResource& resource : report.Resources) {
            bool ok = false;

            for (const TString& nameToFind : names) {
                if (resource.Name.Name.Contains(nameToFind)) {
                    ok = true;
                    break;
                }
            }

            if (!ok) {
                continue;
            }

            /*if (Find(names.begin(), names.end(), resource.Name.Name) == names.end()) {
                continue;
            }*/

            Cout << "Resource '" << resource.Name.Name << "' agent " << report.Agent.Host << " " << report.Agent.Port << " status '" << resource.Status << "'" << Endl;
            agentsToPing.push_back(report.Agent.Host);

            resourcesNotFound.erase(resource.Name.Name);
        }
    }

    for (const TString& str : resourcesNotFound) {
        Cout << "WARNING: resource not found at all: " << str << Endl;
    }

    TVector<TString> uniqueHosts = agentsToPing;
    SortUnique(uniqueHosts);
    Cout << "Hosts: ";
    for (auto it : uniqueHosts) {
        Cout << it << " ";
    }
    Cout << Endl;

    for (const TString& agent : agentsToPing) {
        TVector<TString> agentParts = StringSplitter(agent).Split('-').ToList<TString>();
        TString agentNoMtn = agentParts[0] + "-" + agentParts[1] + ".search.yandex.net";

        Cout << "Agent " << agent << " status (" << agentNoMtn << "):" << Endl;

        system(("ssh " + agentNoMtn + " 'ps -aux | grep skybone-dl'").data());
        Cout << Endl;
    }
}

void AnalyzeAgentsSpace(const TReportsSet& reportsSet) {
    for (const TReport& report : reportsSet.Reports) {
        i64 FreeSpaceAfterDownloads = report.FreeSpace;
        for (const TResource& resource : report.Resources) {
            if (resource.Status == EResourceStatus::IDLE) {
                TTrackerEntry* entry = GlobalLazyTracker.GetEntryByName(resource.Name.Name);
                if (entry) {
                    FreeSpaceAfterDownloads -= entry->GetSize();
                }
            }
        }

        if (FreeSpaceAfterDownloads <= 1_GBs) {
            Cout << "Host: " << report.Agent.Host << " space after downloads: " << FreeSpaceAfterDownloads / 1024.0 / 1024.0 / 1024.0 << " GB" << Endl;
            //system((TStringBuilder() << "ssh " << report.Agent.Host << " df /ssd").data());
        }
    }
}

TString GetPhysicalHost(TString hostName) {
    TVector<TString> parts = StringSplitter(hostName).Split('-').ToList<TString>();
    return parts[0] + "-" + parts[1];
}

int AnalyzeBuilders(int argc, const char** argv) {
    Y_UNUSED(argc, argv);

    /* Report(vla1-8868-970-vla-web-tier1-build-13852.gencfg-c.yandex.net, 13852, ['a_geo_vla', 'a_topology_version-stable-135-r2480', 'cgset_memory_recharge_on_pgfault_1', 'a_metaprj_web', 'itag_copy_on_ssd', 'a_ctype_prod', 'a_prj_web-jupiter', 'a_dc_vla', 'enable_hq_report', 'a_line_vla-03', 'a_tier_none', 'a_itype_builder', 'a_topology_cgset-memory.limit_in_bytes=42949672960', 'enable_hq_poll', 'VLA_WEB_TIER1_BUILD', 'a_topology_group-VLA_WEB_TIER1_BUILD', 'a_topology_cgset-memory.low_limit_in_bytes=42949672960', 'a_prj_web-tier1', 'a_topology_stable-135-r2480'], 2019-11-10 15:42:28)  */

    const TString timestamp = "1601241275";

    TReportContainerMessage data = FetchCached("builders_tier1");
    Cerr << "Report container contains " << data.reportsSize() << " reports" << Endl;

    THashSet<TString> shardsBuilt;

    const ui64 aliveThresholdMinutes = 30;

    ui64 minTimestamp = 0;
    if (aliveThresholdMinutes > 0) {
        minTimestamp = TInstant::Now().TimeT();
        if (minTimestamp >= aliveThresholdMinutes * 60) {
            minTimestamp -= aliveThresholdMinutes * 60;
        } else {
            minTimestamp = 0;
        }
    }

    for (auto reportMessage : data.reports()) {
        const TString& reportsData = reportMessage.data();

        if (reportMessage.timestamp() < minTimestamp) {
            Cerr << "Report from " << reportMessage.Gethost() << " skipped due to timestamp filter: " << reportMessage.timestamp() << " vs " << minTimestamp << Endl;
            continue;
        }

        msgpack::unpacked msg;
        msgpack::unpack(msg, reportsData.data(), reportsData.length());

        msgpack::object obj = msg.get();
        NJson::TJsonValue jsonValue;
        NMsgpack2Json::Msgpack2Json(obj, &jsonValue);

        for (auto it : jsonValue["shards"].GetMap()) {
            if (it.first.Contains(timestamp)) {
                if (it.second["status"].GetString() == "BUILD") {
                    shardsBuilt.insert(it.first);
                } else {
                    Cerr << "Shard " << it.first << " is in state " << it.second["status"].GetString() << Endl;
                }
            }
        }
    }

    Cerr << shardsBuilt.size() << " shards are being built" << Endl;

    for (size_t i = 0; i < 1476; ++i) {
        TString name = "primus-WebTier1-0-" + ToString(i) + "-" + timestamp;
        if (!shardsBuilt.count(name)) {
            Cerr << "Shard " << name << " is not being built anywhere" << Endl;
        }
    }

    return 0;

    TReportsSet reportsSet = ParseReports(data, 0);
    Cerr << "Reports set contains " << reportsSet.Reports.size() << " reports" << Endl;

    SortBy(reportsSet.Reports, [](const TReport& report) {
        return report.FreeSpace;
    });

    for (auto it : reportsSet.Reports.back().Resources) {
        Cerr << it.Name.Name << Endl;
    }

    return 0;

    for (size_t i = 0; i < Min<size_t>(10000, reportsSet.Reports.size()); ++i) {
        const TReport& report = reportsSet.Reports[i];
        Cerr << report.Agent.Host << " " << report.FreeSpace / 1024.0 / 1024.0 / 1024.0 << " Gb free" << Endl;
    }
    return 0;
}

void ShowControllerDeployStatus(const TConfig& config, TConstArrayRef<TString> locations) {
    NYT::IClientPtr client = NYT::CreateClient("arnold");

    THashMap<TString, TClusterConfig> clusters;
    for (auto&& cluster : config.GetClusters().GetCluster()) {
        clusters[cluster.GetName()] = cluster;
    }

    TVector<TStringBuf> notReady;
    for (TStringBuf location : locations) {
        const TClusterConfig* config = clusters.FindPtr(location);
        if (!config) {
            Cerr << NColorizer::YELLOW << "Warning: unknown cluster " << location << NColorizer::RESET << Endl;
            continue;
        }

        auto rows = client->SelectRows("* FROM [" + config->GetStatus() + "]");
        if (rows.empty()) {
            Cerr << NColorizer::YELLOW << "Warning: empty status for " << location << NColorizer::RESET << Endl;
            continue;
        }

        NYT::TNode status = *MaxElementBy(rows, [](const NYT::TNode& node) {
            return node["Id"].AsInt64();
        });

        Cout
            << "Status for " << location << ": "
            << NYT::NodeToCanonicalYsonString(status) << Endl;

        TVector<TStringBuf> states = StringSplitter(status["Deploy"].AsString()).Split(' ');
        if (Find(states, GlobalTimestamp) == states.end()) {
            notReady.emplace_back(location);
        }
    }

    if (not notReady.empty()) {
        Cout << NColorizer::RED << "Not ready locations:";
        for (TStringBuf location : notReady) {
            Cout << ' ' << location;
        }
        Cout << NColorizer::RESET << Endl;
    }
}

int AnalyzeDeployment(int argc, const char** argv) {
    TShowStatsLimits showStatsLimits;

    TVector<TString> locations;
    TVector<TString> resourcesToFind;

    ui32 aliveThresholdMinutes = 0;
    TString outputFileName;

    NLastGetopt::TOpts opts;
    opts.AddLongOption("timestamp", "Timestamp").Required().RequiredArgument("<ypath>").StoreResult(&GlobalTimestamp);
    opts.AddLongOption("location", "Add location to analyze").Optional().RequiredArgument("<ypath>").AppendTo(&locations);
    opts.AddLongOption("find-resource", "Add resource to find").Optional().RequiredArgument("<ypath>").AppendTo(&resourcesToFind);
    opts.AddLongOption("no-cache", "Disable (overwrite existing) cache").Optional().NoArgument().StoreValue(&GlobalCacheEnabled, false);

    opts.AddLongOption("alive-threshold",
        "Too old time to ignore reports, in minutes. This is exactly what controller does when determining target deployed state. Will affect all further statistics, find resource etc. Will not correctly work with cache.")
        .Optional().DefaultValue(30).RequiredArgument("<minutes>").StoreResult(&aliveThresholdMinutes);

    opts.AddLongOption("max-local-replics", "Max number of shards low on local replics to print")
        .Optional().RequiredArgument("<ypath>").DefaultValue(showStatsLimits.MaxShardsByLocalReplics).StoreResult(&showStatsLimits.MaxShardsByLocalReplics);
    opts.AddLongOption("max-not-in-tracker", "Max number of local shards not in tracker to print")
        .Optional().RequiredArgument("<ypath>").DefaultValue(showStatsLimits.MaxShardsNotInTracker).StoreResult(&showStatsLimits.MaxShardsNotInTracker);
    opts.AddLongOption("max-chunks", "Max number of chunks low on parts to print")
        .Optional().RequiredArgument("<ypath>").DefaultValue(showStatsLimits.MaxChunksByPartsCount).StoreResult(&showStatsLimits.MaxChunksByPartsCount);
    opts.AddLongOption("max-deployers", "Max number of deployers with not ready resources to print")
        .Optional().RequiredArgument("<ypath>").DefaultValue(showStatsLimits.MaxDeployersWithNotReadyResources).StoreResult(&showStatsLimits.MaxDeployersWithNotReadyResources);
    opts.AddLongOption("output-file", "Save stats in json-serialized protobuf format").Optional().RequiredArgument("<filename>").StoreResult(&outputFileName);

    NLastGetopt::TOptsParseResult parseResult(&opts, argc, argv);

    Y_ENSURE(!locations.empty(), "--location cannot be empty (could be: pip, sas, man, vla)");

    ui64 minTimestamp = 0;
    if (aliveThresholdMinutes > 0) {
        minTimestamp = TInstant::Now().TimeT();
        if (minTimestamp >= aliveThresholdMinutes * 60) {
            minTimestamp -= aliveThresholdMinutes * 60;
        } else {
            minTimestamp = 0;
        }

        Cout << "Ignoring reports older than " << aliveThresholdMinutes << " minutes" << Endl;
    }

    TConfig config = ParseFromTextFormat<TConfig>("config.txt");
    ShowControllerDeployStatus(config, locations);

    for (const TString& location : locations) {
        Cout << NColorizer::YELLOW << "=== Location " << location << " ===" << NColorizer::RESET << Endl;

        TReportContainerMessage data = FetchCached(location);

        TReportsSet reportsSet = ParseReports(data, minTimestamp);

        TDeploymentStats stats = ShowStats(config, reportsSet, location, showStatsLimits);
        Cout << Endl;

        if (!resourcesToFind.empty()) {
            FindResources(reportsSet, resourcesToFind);
        }

        if (outputFileName) {
            TFileOutput output(outputFileName);
            output.Write(ProtoToJson(stats));
        }
    }

    return 0;
}

using TReportsProcessor = std::function<void (const TReportsSet&, const TString& timestamp, const TString& location, ui64 topSize)>;

int ProcessReports(int argc, const char** argv, const TReportsProcessor& reportsProcessor) {
    TVector<TString> locations;
    ui64 topSize = 10;

    NLastGetopt::TOpts opts;
    opts.AddLongOption("timestamp", "Timestamp").Required().RequiredArgument("<ypath>").StoreResult(&GlobalTimestamp);
    opts.AddLongOption("location", "Add location to analyze").Optional().RequiredArgument("<ypath>").AppendTo(&locations);
    opts.AddLongOption("no-cache", "Disable (overwrite existing) cache").Optional().NoArgument().StoreValue(&GlobalCacheEnabled, false);
    opts.AddLongOption("top-size", "top size").Optional().RequiredArgument("<ui64>").DefaultValue(topSize).StoreResult(&topSize);

    NLastGetopt::TOptsParseResult parseResult(&opts, argc, argv);

    Y_ENSURE(!locations.empty(), "--location cannot be empty (could be: pip, sas, man, vla)");

    for (const TString& location : locations) {
        TReportContainerMessage data = FetchCached(location);

        TReportsSet reportsSet = ParseReports(data, 0);

        if (!reportsSet.Reports.empty()) {
            reportsProcessor(reportsSet, GlobalTimestamp, location, topSize);
        }
    }

    return 0;
}

int DumpReportsMode(int argc, const char** argv) {
    return ProcessReports(argc, argv, DumpReportsSet);
}

int ShowTopUnpreparedShardsMode(int argc, const char** argv) {
    return ProcessReports(argc, argv, ShowTopUnpreparedShards);
}

int ShowTopUnpreparedHostsMode(int argc, const char** argv) {
    return ProcessReports(argc, argv, ShowTopUnpreparedHosts);
}

TString TimestampToDatetimeString(const ui32 timestamp) {
    time_t timeValue = timestamp;
    std::tm tmStruct = *std::localtime(&timeValue);
    return Strftime("%F %T", &tmStruct);
}

int AnalyzeRelocation(int argc, const char** argv) {
    TVector<TString> locations;
    TVector<TString> timestamps;
    ui32 numTimestamps = 2;

    NLastGetopt::TOpts opts;
    opts.AddLongOption("timestamp", "Add timestamp to analyze (overrides num-timestamps)").Optional().RequiredArgument("<ypath>").AppendTo(&timestamps);
    opts.AddLongOption("num-timestamps", "Num last mappings to analyze").Optional().RequiredArgument("<ypath>").StoreResult(&numTimestamps).DefaultValue(2);
    opts.AddLongOption("location", "Add location to analyze").Optional().RequiredArgument("<ypath>").AppendTo(&locations);
    NLastGetopt::TOptsParseResult parseResult(&opts, argc, argv);

    NYT::IClientPtr client = NYT::CreateClient("arnold");

    if (timestamps.empty()) {
        TVector<NYT::TNode> tables = client->List("//home/cajuper/user/web/prod/chunks/pip/WebTier1/mapping");
        for (auto it : tables) {
            timestamps.push_back(it.AsString());
        }
        Sort(timestamps);

        if (timestamps.size() > numTimestamps) {
            timestamps.erase(timestamps.begin(), timestamps.begin() + (timestamps.size() - numTimestamps));
        }
    }

    Cerr << "Will analyze timestamps: ";
    for (auto it : timestamps) {
        Cerr << it << " ";
    }
    Cerr << Endl;

    Y_ENSURE(timestamps.size() >= 2);
    Y_ENSURE(!locations.empty());

    THashMap<std::pair<TString, TString>, THashMap<TString, TVector<TString>>> partToHostByLocationTimestamp;

    auto dropTimestamp = [](const TString& shardName) {
        TVector<TString> shardNameParts;
        StringSplitter(shardName).SplitByString("-").SkipEmpty().Collect(&shardNameParts);

        TString timelassShardName;
        for (size_t i = 0; i < shardNameParts.size() - 1; ++i) {
            timelassShardName += shardNameParts[i];
            if (i < shardNameParts.size() - 2) {
                timelassShardName += "-";
            }
        }

        return timelassShardName;
    };

    auto loadMapping = [&client, &partToHostByLocationTimestamp, &dropTimestamp](const TString& location, const TString& timestamp) {
        if (partToHostByLocationTimestamp.contains(std::make_pair(location, timestamp))) {
            return partToHostByLocationTimestamp[std::make_pair(location, timestamp)];
        }

        THashMap<TString, TVector<TString>> partToHost;

        TString mappingTablePath = TStringBuilder() << "//home/cajuper/user/web/prod/chunks/" << location << "/WebTier1/mapping/" + timestamp;
        NYT::TTableReaderPtr<TMappingEntry> reader = client->CreateTableReader<TMappingEntry>(mappingTablePath);

        for (; reader->IsValid(); reader->Next()) {
            const TMappingEntry& entry = reader->GetRow();
            const TString name = dropTimestamp(entry.GetShard()) + "/" + entry.GetPath();

            // Y_ENSURE(!partToHost.contains(name), "Duplicate part: " << name);
            partToHost[name].push_back(entry.GetHost());
        }

        //Cerr << "Loaded mapping for " << timestamp << ": " << partToHost.size() << " parts" << Endl;

        partToHostByLocationTimestamp[std::make_pair(location, timestamp)] = partToHost;
        return partToHost;
    };

    Sort(timestamps);

    auto invalidateTimestampCache = [&partToHostByLocationTimestamp](const TString& timestamp) {
        TVector<std::pair<TString, TString>> keysToErase;

        for (const auto& it : partToHostByLocationTimestamp) {
            if (it.first.second == timestamp) {
                keysToErase.push_back(it.first);
            }
        }

        for (auto it : keysToErase) {
            partToHostByLocationTimestamp.erase(it);
        }
    };

    for (size_t i = 0; i + 1 < timestamps.size(); ++i) {
        if (i > 0) {
            invalidateTimestampCache(timestamps[i - 1]);
        }

        Cerr << "=== Part relocation stats "
            << TimestampToDatetimeString(FromString<ui32>(timestamps[i])) << " => " << TimestampToDatetimeString(FromString<ui32>(timestamps[i + 1]))
             << " (" << timestamps[i] << " => " << timestamps[i + 1] << ") "
            << Endl;

        for (const TString& location : locations) {
            auto prevMapping = loadMapping(location, timestamps[i]);
            auto curMapping = loadMapping(location, timestamps[i + 1]);

            ui32 partsRelocated = 0;
            ui32 partsRemain = 0;
            ui32 partsAdded = 0;
            ui32 partsDeleted = 0;

            for (auto it : curMapping) {
                if (!prevMapping.contains(it.first)) {
                    partsAdded += curMapping[it.first].size();
                } else {
                    TVector<TString> mappingIntersection;
                    std::set_intersection(
                        curMapping[it.first].begin(), curMapping[it.first].end(),
                        prevMapping[it.first].begin(), prevMapping[it.first].end(),
                        std::back_inserter(mappingIntersection));

                    if (curMapping[it.first].size() < prevMapping[it.first].size()) {
                        partsDeleted += prevMapping[it.first].size() - curMapping[it.first].size();
                    } else if (curMapping[it.first].size() > prevMapping[it.first].size()) {
                        partsAdded += curMapping[it.first].size() - prevMapping[it.first].size();
                    }

                    size_t commonSize = std::min(curMapping[it.first].size(), prevMapping[it.first].size());
                    if (mappingIntersection.size() < commonSize) {
                        partsRelocated += commonSize - mappingIntersection.size();
                    }
                    partsRemain += mappingIntersection.size();
                }
            }

            for (auto it : prevMapping) {
                if (!curMapping.contains(it.first)) {
                    partsDeleted += prevMapping[it.first].size();
                }
            }

            Cerr << "Location '" << location
                << "' need deploy: " << partsRelocated + partsAdded << " (" << Percents(partsRelocated + partsAdded, partsRelocated + partsRemain) << "% from prev) "
                << ", relocate: " << partsRelocated << " (" << Percents(partsRelocated, partsRelocated + partsRemain) << "%) "
                << ", remain: " << partsRemain
                << ", added: " << partsAdded
                << ", deleted: " << partsDeleted
                << Endl;
        }

        Cerr << Endl;
    }

    return 0;
}

int Main(int argc, const char** argv) {
    NYT::Initialize(argc, argv);

    TModChooser modChooser;
    modChooser.AddMode("AnalyzeDeployment", AnalyzeDeployment, "Analyze deployment of base in locations");
    modChooser.AddMode("AnalyzeRelocation", AnalyzeRelocation, "Analyze relocations (movement to differnet hosts) of chunks between states due to planning");
    modChooser.AddMode("AnalyzeBuilders", AnalyzeBuilders, "Analyze free space on builders");
    modChooser.AddMode("DumpReports", DumpReportsMode, "Dump tracker reports (in tskv format)");
    modChooser.AddMode("ShowTopUnpreparedShards", ShowTopUnpreparedShardsMode, "Show top unprepared shards(chunks) for each tier");
    modChooser.AddMode("ShowTopUnpreparedHosts", ShowTopUnpreparedHostsMode, "Show top unprepared hosts");

    return modChooser.Run(argc, argv);
}

} // namespace NInfra::NDeployViewer

int main(int argc, const char** argv) {
    return NInfra::NDeployViewer::Main(argc, argv);
}
