# coding: utf8
import datetime
import time
from argparse import ArgumentParser


TEMPLATE = """Chunks: {{
  Shard: {shard_number}
  Version: {version}
  Id: {chunk}
  Part: 0
  Index: "{index}"
  Tier: {tier}
  Path: "{chunk}/0/{index}/{part_filename}"
}}"""

INDEX_TO_FILENAME = {
    'arc': 'wad.part',
    'extarc': 'extinfo.part',
    'keyinv': 'keyinv.part',
    'common': 'common.part',
}

VIDEO_INDEX_TO_FILENAME = {
    'arc': 'wad.part',
    'extarc': 'extinfo.part',
    's2t': 's2t.part',
    's2tdssm': 's2tdssm.part',
    'omni': 'omni.part',
}


def make_config(tier, shard_number, version, chunks_list, index_list):
    part_configs = []
    for chunk in chunks_list:
        for index, part_filename in index_list.items():
            part_config = TEMPLATE.format(
                shard_number=shard_number,
                version=version,
                chunk=chunk,
                index=index,
                part_filename=part_filename,
                tier=tier,
            )
            part_configs.append(part_config)
    return part_configs


def parse_cmd():
    parser = ArgumentParser(description='Script to generate remote storage cache config.')

    parser.add_argument('--shard-name', type=str, required=True,
                        help='Shard name')

    return parser.parse_args()


def convert_datetime_to_version(datetime_str):
    dt = datetime.datetime.strptime(datetime_str, '%Y%m%d-%H%M%S')
    unixtime = time.mktime(dt.timetuple())
    return int(unixtime)


def main():
    shard_name = parse_cmd().shard_name

    shard_patrs = shard_name.split('-')
    index_list = INDEX_TO_FILENAME

    if 'WebTier0' in shard_name:
        shard_number = shard_patrs[3]
        version = shard_patrs[-1]
        tier = 'WebTier0'
        chunks_list = range(1476, 1476 * 2)
    elif 'WebTier1' in shard_name:
        shard_number = shard_patrs[3]
        version = shard_patrs[-1]
        tier = 'WebTier1'
        chunks_list = range(72, 72 * 2)
    elif 'vidsidxpt' in shard_name:
        shard_number = int(shard_patrs[1])
        version = convert_datetime_to_version('{0}-{1}'.format(shard_patrs[2], shard_patrs[3]))
        tier = 'VideoPlatinum'
        chunks_list = range(13, 13 * 2)
        index_list = VIDEO_INDEX_TO_FILENAME
    elif 'imgsidx' in shard_name:
        shard_number = int(shard_patrs[1])
        version = convert_datetime_to_version('{0}-{1}'.format(shard_patrs[2], shard_patrs[3]))
        tier = 'ImgTier0'
        chunks_list = range(32, 32 * 2)
    else:
        raise RuntimeError('Tier for shard {} is not supported'.format(shard_name))

    part_configs = make_config(tier, shard_number, version, chunks_list, index_list)
    config = '\n'.join(part_configs)
    print(config)


if __name__ == '__main__':
    main()
