import os
import logging
import shutil

import infra.callisto.libraries.warmup.vmtouch as vmtouch
import infra.callisto.libraries.warmup as warmup


ACTIONS = {}


class ActionMetaClass(type):
    def __new__(mcs, name, bases, dct):
        class_ = super(ActionMetaClass, mcs).__new__(mcs, name, bases, dct)
        if class_.cmd is not None:
            ACTIONS[class_.cmd] = class_
        return class_


class Action(object):
    cmd = None
    __metaclass__ = ActionMetaClass

    @classmethod
    def add_args(cls, parser):
        pass

    def execute(self, environment, args):
        raise NotImplementedError()


class ShardName(Action):
    cmd = 'shard_name'

    def execute(self, environment, args):
        print environment.shard_name()


class CopyOnSSD(Action):
    cmd = 'copy_on_ssd'

    def execute(self, environment, args):
        exit(int(not environment.copy_on_ssd()))


class StorageRoot(Action):
    cmd = 'storage_root'

    def execute(self, environment, args):
        print environment.storage_root()


class InstanceDir(Action):
    cmd = 'instance_dir'

    def execute(self, environment, args):
        print environment.instance_dir()


class DbTimestamp(Action):
    cmd = 'db_timestamp'

    def execute(self, environment, args):
        print environment.db_timestamp()


class TierName(Action):
    cmd = 'tier_name'

    def execute(self, environment, args):
        print environment.tier_name()


class Group(Action):
    cmd = 'group'

    def execute(self, environment, args):
        print environment.group()


class ShardNumber(Action):
    cmd = 'shard_number'

    def execute(self, environment, args):
        print environment.shard_number()


class IsWarmed(Action):
    cmd = 'is_warmed'

    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--path', required=True, type=str)
        parser.add_argument('--threshold', required=True, type=float)

    def execute(self, environment, args):
        assert os.path.exists(args.path)
        warmness = vmtouch.in_memory_percent(args.path)
        logging.info('path: %s, warmness: %s, threshold: %s', args.path, warmness, args.threshold)
        if warmness >= args.threshold:
            exit(0)
        else:
            exit(1)


class WarmupShard(Action):
    cmd = 'warmup_shard'

    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--port', required=True, type=int)
        parser.add_argument('--path', required=True, type=str)
        parser.add_argument('--threshold', required=True, type=float)
        parser.add_argument('--dolbilo-path', required=True, type=str)
        parser.add_argument('--wood-path', required=True, type=str)

    def execute(self, environment, args):
        heater = warmup.DolbiloHeater(
            environment.hostname(),
            args.port,
            args.dolbilo_path,
            args.wood_path,
        )
        warmup.Supervisor(heater, args.threshold, args.path).start().join()


class LinkShard(Action):
    cmd = 'link_shard'

    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--mode', required=True, choices=['shardtool', 'deployer'])

    def execute(self, environment, args):
        destination = os.path.join(environment.instance_dir(), environment.shard_name())
        if args.mode == 'shardtool':
            source = os.path.join(environment.storage_root(), environment.shard_name())
        else:
            source = environment.deployer_shard_path()
        if os.path.exists(source):
            _set_symlink(source, destination)
            logging.info('linked %s to %s', source, destination)
            exit(0)
        else:
            logging.info('source %s does not exist', source)
            exit(1)


class PathToNamespace(Action):
    cmd = 'path_to_namespace'

    def execute(self, environment, args):
        print environment.deployer_namespace_path(environment.tier_name())


class ConfigName(Action):
    cmd = 'config_name'

    def execute(self, environment, args):
        print environment.config_name()


class RsConfigName(Action):
    cmd = 'rs_config_name'

    def execute(self, environment, args):
        print environment.rs_config_name()


class SetTopologyDir(Action):
    cmd = 'set_topology_dir'

    @classmethod
    def add_args(cls, parser):
        parser.add_argument('--target', required=True)
        parser.add_argument('--storage', required=True)
        parser.add_argument('--rotate-limit', default=10, type=int)

    def execute(self, environment, args):
        timestamp = str(int(environment.db_timestamp()))
        current_dir = os.path.join(args.storage, timestamp)

        if not os.path.exists(current_dir):
            os.makedirs(current_dir)

        _set_symlink(current_dir, args.target)

        def parse_ts(file):
            try:
                return int(file)
            except ValueError:
                return 0

        dirs = os.listdir(args.storage)
        dirs.sort(key=parse_ts)

        for file in dirs[:-args.rotate_limit]:
            if timestamp != file:
                path = os.path.join(args.storage, file)
                print('removing ' + path)
                shutil.rmtree(path)


def _set_symlink(source, destination):
    if os.path.islink(destination) and os.readlink(destination) == source:
        return
    if os.path.islink(destination):
        os.unlink(destination)
    os.symlink(source, destination)
