import datetime
import os

import infra.callisto.deploy.storage.storage as resources_storage
import infra.callisto.deploy.resource as deploy_resource


class Env(object):
    def __init__(self, vars_file_name):
        self.vars_file_name = vars_file_name

    @staticmethod
    def instance_dir():
        return os.environ['BSCONFIG_IDIR']

    @staticmethod
    def hostname():
        return os.environ['BSCONFIG_IHOST']

    @staticmethod
    def node_name():
        return os.environ['NODE_NAME']

    def storage_root(self):
        raise NotImplementedError()

    def shard_name(self):
        raise NotImplementedError()

    def shard_number(self):
        raise NotImplementedError()

    def tier_name(self):
        raise NotImplementedError()

    def copy_on_ssd(self):
        raise NotImplementedError()

    def config_name(self):
        raise NotImplementedError()

    def db_timestamp(self):
        vars_conf = self.env_file(self.vars_file_name)
        return '{:010}'.format(int(vars_conf['DB_TIMESTAMP']))

    def env_file(self, file_name):
        file_path = os.path.join(self.instance_dir(), file_name)
        if not os.path.exists(file_path):
            raise RuntimeError('{} does not exist in {}'.format(file_name, self.instance_dir()))
        with open(file_path) as fd:
            content = fd.read()
        return _parse_env_file(content)

    def deployer_namespace_path(self, tier_name):
        vars_conf = self.env_file(self.vars_file_name)
        namespace = deploy_resource.safe_join_path(vars_conf['NAMESPACE_PREFIX'], vars_conf['DB_TIMESTAMP'], tier_name)
        return resources_storage.namespace_path(
            self.storage_root(),
            namespace,
        )

    def _shard_name_for_path(self):
        name = self.shard_name()
        if self.tier_name() in ('AttributeWebTier0', 'WebTier0', 'WebTier1', 'KeyInvWebTier0'):
            name += '/local'
        return name

    def deployer_shard_path(self):
        vars_conf = self.env_file(self.vars_file_name)
        namespace = deploy_resource.safe_join_path(vars_conf['NAMESPACE_PREFIX'], vars_conf['DB_TIMESTAMP'],
                                                   self.tier_name())
        name = self._shard_name_for_path()
        return resources_storage.resource_path(
            self.storage_root(),
            namespace,
            name,
        )


def timestamp_to_yt_state(ts):
    return datetime.datetime.fromtimestamp(int(ts)).strftime('%Y%m%d-%H%M%S')


def _parse_env_file(text):
    text = text.replace('export ', '')
    text = text.split('\n')

    result = {}
    for line in text:
        if line.count('=') == 1:
            key, value = line.split('=')
            result[key.strip()] = value.strip().strip('\"')
    return result
