import os
import requests
import socket

import common

import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.deploy.storage.storage as resources_storage
import infra.callisto.deploy.resource as deploy_resource


class Env(common.Env):
    def __init__(self, vars_file_name):
        super(Env, self).__init__(vars_file_name)

    @staticmethod
    def instance_dir():
        return '.'

    @staticmethod
    def hostname():
        return socket.getfqdn()

    @staticmethod
    def node_name():
        return os.environ['PORTO_HOST']

    def db_timestamp(self):
        vars_conf = self.env_file(self.vars_file_name)
        return '{:010}'.format(int(vars_conf['DB_TIMESTAMP']))

    def env_file(self, file_name):
        file_path = os.path.join(self.instance_dir(), file_name, file_name)  # sic!
        if not os.path.exists(file_path):
            raise RuntimeError('{} does not exist in {}'.format(file_name, self.instance_dir()))
        with open(file_path) as fd:
            content = fd.read()
        return common._parse_env_file(content)

    def deployer_namespace_path(self, tier_name):
        vars_conf = self.env_file(self.vars_file_name)
        namespace = deploy_resource.safe_join_path(vars_conf['NAMESPACE_PREFIX'], vars_conf['DB_TIMESTAMP'], tier_name)
        return resources_storage.namespace_path(self.storage_root(), namespace)

    def deployer_shard_path(self):
        vars_conf = self.env_file(self.vars_file_name)
        namespace = deploy_resource.safe_join_path(vars_conf['NAMESPACE_PREFIX'], vars_conf['DB_TIMESTAMP'],
                                                   self.tier_name())
        name = self._shard_name_for_path()
        return resources_storage.resource_path(self.storage_root(), namespace, name)

    def shard_name(self):
        tier = tiers.TIERS[self.tier_name()]

        return tier.list_shards(
            self.db_timestamp()
        )[self.shard_number()].fullname

    def tier_name(self):
        return os.environ['GENCFG_TIER']

    def shard_count(self):
        if 'GENCFG_SHARD_COUNT' in os.environ:
            return int(os.environ['GENCFG_SHARD_COUNT'])
        else:
            return tiers.TIERS[self.tier_name()].shards_count

    def copy_on_ssd(self):
        return True

    def storage_root(self):
        generation_conf = self.env_file('generation.conf')
        storage_root = generation_conf['GENERATION']

        return storage_root if os.path.isabs(storage_root) else os.path.join('/', storage_root)

    def config_name(self):
        return '{}.{}.{}.cfg'.format(os.environ['DEPLOY_STAGE_ID'], os.environ['DEPLOY_UNIT_ID'], self.shard_number())

    def rs_config_name(self):
        if self.tier_name() in ('WebTier0', 'WebTier1', 'VideoPlatinum'):
            return '{}.proto'.format(self.shard_name())
        if self.tier_name() == 'AttributeWebTier0':
            return '{}.proto'.format(self.shard_name()).replace('AttributeWebTier0', 'WebTier0')
        raise NotImplementedError('Other tiers not supported')

    def shard_number(self):
        return int(_labels()['pod_index']) % self.shard_count()


def _labels():
    return requests.get('http://localhost:1/pod_attributes').json()['metadata']['labels']
