#! /usr/bin/python
# -*- coding: utf-8 -*-
import aniso8601
import collections
import datetime
import logging
import math
import os
import requests

import solomon

import infra.callisto.controllers.sdk.tier as tiers
import infra.callisto.controllers.user.jupiter.sas as web_sas
import infra.callisto.controllers.utils.funcs as funcs
import infra.callisto.libraries.yt as yt_utils


TierRequirements = collections.namedtuple('TierRequirements', ['tier', 'unique_ratio', 'total_ratio'])
Table = collections.namedtuple('Table', ['proxy', 'path'])
Source = collections.namedtuple('Source', ['name', 'table', 'tiers'])

PlatinumTier0 = TierRequirements(tiers.PlatinumTier0, 1.0, 0.95)
WebTier1 = TierRequirements(tiers.WebTier1, 0.99, 0.90)
MsuseardataJupiterTier0 = TierRequirements(tiers.MsUserData, 1.0, 0.6)
GeminiTier = TierRequirements(tiers.GeminiTier, 0.99, 0.90)
CastorTier = TierRequirements(tiers.CastorTier, 1.0, 0.90)

SOURCES = [
    Source(
        name='ProductionBase',
        table=Table('arnold', '//home/jupiter/shard_deploy/production_base'),
        tiers=(PlatinumTier0, WebTier1),
    ),
    Source(
        name='ProductionMiddle',
        table=Table('arnold', '//home/jupiter/shard_deploy/production_middlesearch_shard'),
        tiers=(MsuseardataJupiterTier0,),
    ),
    Source(
        name='GeminiBase',
        table=Table('arnold', '//home/jupiter/shard_deploy/gemini_base'),
        tiers=(GeminiTier, CastorTier),
    ),
]

TRACKER_PATH_PREFIX = '//home/cajuper/tracker/'
DEPLOY_URLS = [
    'http://ctrl.clusterstate.yandex-team.ru/web/prod-man/view/deploy_namespace_percentage',
    'http://ctrl.clusterstate.yandex-team.ru/web/prod-sas/view/deploy_namespace_percentage',
    'http://ctrl.clusterstate.yandex-team.ru/web/prod-vla/view/deploy_namespace_percentage',
]


def make_solomon_client(token):
    return solomon.BasePushApiReporter(
        project='jupiter',
        cluster='prod',
        service='metrics',
        url='http://solomon.yandex.net',
        auth_provider=solomon.OAuthProvider(token),
    )


def deploy_start_time(timestamp):
    sas_table = web_sas.get_yt_target_table(readonly=True)
    for target_wrapper in reversed(sas_table.get_last_n(5)):
        if int(timestamp) in target_wrapper.target.deploy:
            return target_wrapper.time
    return None


def current_prod_timestamp():
    response = requests.get(
        'http://ctrl.clusterstate.yandex-team.ru/web/prod/jupiter/oldprod',
        timeout=30,
    )
    response.raise_for_status()
    common_state = response.json()['common']['yt_state']
    assert common_state is not None
    return int(common_state)


def next_prod_timestamp(source, prod_timestamp):
    yt_client = yt_utils.create_yt_client(source.table.proxy)
    for row in yt_client.read_table(source.table.path, format='json'):
        if funcs.yt_state_to_timestamp(row['State']) > prod_timestamp:
            return funcs.yt_state_to_timestamp(row['State'])
    return None


def current_build_timestamp(source_table):
    yt_client = yt_utils.create_yt_client(source_table.proxy)
    for row in reversed(list(yt_client.read_table(source_table.path, format='json'))):
        return funcs.yt_state_to_timestamp(row['State']), aniso8601.parse_datetime(row['Time']).replace(tzinfo=None)
    raise RuntimeError('table {} is empty!'.format(source_table))


def registered_shards_count(tier_name, timestamp):
    path = os.path.join(TRACKER_PATH_PREFIX, 'web/prod', str(timestamp), tier_name, 'table')
    yt_client = yt_utils.create_yt_client('arnold')
    if yt_client.exists(path):
        resources = yt_client.select_rows('name from [{}] where not is_substr("/", name)'.format(path), format='json')
        return sum(1 for _ in resources)
    return 0


def build_is_done(source, timestamp):
    for tier_requirements in source.tiers:
        count = registered_shards_count(tier_requirements.tier.name, timestamp)
        if count < math.floor(tier_requirements.tier.shards_count * tier_requirements.unique_ratio):
            return False
    return True


def _get_tier_timestamp(deploy_percentage, tier_name, timestamp):
    for namespace, state in deploy_percentage.items():
        if '/{}/{}/'.format(timestamp, tier_name) in namespace:
            return state
    return None


def deploy_tier_is_ready(tier_requirements, deploy_percentage_list, timestamp):
    for deploy_percentage in deploy_percentage_list:
        state = _get_tier_timestamp(deploy_percentage, tier_requirements.tier.name, timestamp)
        if state and (
            state['resources']['percentage'] < tier_requirements.unique_ratio
            or state['replicas']['percentage'] < tier_requirements.total_ratio
        ):
            return False
    return True


def deploy_is_done(source, deploy_percentage_list, timestamp):
    timestamp = str(timestamp)
    for tier in source.tiers:
        if not deploy_tier_is_ready(tier, deploy_percentage_list, timestamp):
            return False
    return True


def push_to_solomon(source_name, deploy_duration, build_duration, token):
    client = make_solomon_client(token)
    client.set_value(
        [source_name + 'DeployDuration', source_name + 'BuildDuration'],
        [deploy_duration, build_duration]
    )


def get_build_duration(source):
    build_timestamp, build_started = current_build_timestamp(source.table)
    build_done = build_is_done(source, build_timestamp)
    return 0 if build_done else (datetime.datetime.now() - build_started).total_seconds() / 3600.0


def get_deploy_duration(source, prod_timestamp, deploy_percentage_list):
    next_prod_timestamp_ = next_prod_timestamp(source, prod_timestamp)
    if next_prod_timestamp_ is None:
        return 0
    deploy_started = deploy_start_time(next_prod_timestamp_)
    if deploy_started is None:
        return 0
    deploy_done = deploy_is_done(source, deploy_percentage_list, next_prod_timestamp_)
    return 0 if deploy_done else (datetime.datetime.now() - deploy_started).total_seconds() / 3600.0


def update_source(source, prod_timestamp, deploy_percentage_list, solomon_token):
    deploy_duration = get_deploy_duration(source, prod_timestamp, deploy_percentage_list)
    build_duration = get_build_duration(source)
    push_to_solomon(
        source.name,
        deploy_duration=deploy_duration,
        build_duration=build_duration,
        token=solomon_token,
    )


def execute(solomon_token):
    deploy_percentage_list = []
    logging.info('fetching deploy status')
    for url in DEPLOY_URLS:
        deploy_percentage_list.append(requests.get(url, timeout=60).json())
    logging.info('fetching prod timestamp')
    prod_timestamp = current_prod_timestamp()
    for source in SOURCES:
        logging.info('updating %s', source.name)
        update_source(source, prod_timestamp, deploy_percentage_list, solomon_token)
    logging.info('done')
